package ru.yandex.crypta.service.staff;

import java.sql.Timestamp;
import java.util.HashMap;
import java.util.List;

import javax.inject.Inject;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.base.Splitter;

import ru.yandex.crypta.clients.staff.StaffClient;
import ru.yandex.crypta.graph.Identifier;
import ru.yandex.crypta.lib.proto.identifiers.EIdType;

public class DefaultStaffService implements StaffService {
    private Timestamp lastUpdate;
    private Timestamp fullUpdateTimestamp;
    private final HashMap<String, String> boundLogins;
    private final StaffClient staffClient;
    private final static long MILLISECONDS_IN_DAY = 86400000;


    @Inject
    DefaultStaffService(StaffClient staffClient) {
        this.staffClient = staffClient;
        this.boundLogins = new HashMap<>();
        this.lastUpdate = null;
        this.fullUpdateTimestamp = null;
    }

    private void addLoginToState(String login, String user) {
        Identifier identifier = new Identifier(EIdType.LOGIN, login);
        boundLogins.put(identifier.getNormalizedValue(), user);
    }

    private void extractAllLogins(JsonNode person) {
        var yandexAccount = person.get("yandex");
        var user = person.get("login").asText();
        var login = yandexAccount.get("login");

        if (login.isTextual()) {
            addLoginToState(login.asText(), user);
        }

        person.get("accounts").forEach(account -> {
            var accountType = account.get("type").asText();
            if (accountType.equals("another_work_email") || accountType.equals("personal_email")) {
                var splitEmail = Splitter.on("@").splitToList(account.get("value").asText());
                if (splitEmail.size() == 2) {
                    var splitDomain = Splitter.on("\\.").splitToList(splitEmail.get(1));
                    if (splitDomain.get(0).equals("ya") || splitDomain.get(0).equals("yandex")) {
                        addLoginToState(splitEmail.get(0), user);
                    }
                }
            }
        });
    }

    @Override
    public synchronized HashMap<String, String> getAllBoundLogins() {
        List<JsonNode> persons;

        if (lastUpdate != null && fullUpdateTimestamp!= null && (lastUpdate.getTime() - fullUpdateTimestamp.getTime()) <= MILLISECONDS_IN_DAY) {
            persons = staffClient.getAllPersons(lastUpdate);
        } else {
            persons = staffClient.getAllPersons();
            fullUpdateTimestamp = new Timestamp(System.currentTimeMillis());
        }

        persons.forEach(this::extractAllLogins);

        lastUpdate = new Timestamp(System.currentTimeMillis());

        return boundLogins;
    }
}
