package ru.yandex.crypta.service.task;

import java.util.List;
import java.util.Map;

import javax.inject.Inject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.crypta.clients.reactor.ReactorClient;
import ru.yandex.crypta.clients.reactor.artifact.ArtifactCreateRequest;
import ru.yandex.crypta.clients.reactor.artifact.ArtifactInstantiationResponse;
import ru.yandex.crypta.clients.reactor.artifact.KeyValueAttributes;
import ru.yandex.crypta.clients.reactor.artifact.LastArtifactInstanceResponse;
import ru.yandex.crypta.clients.reactor.artifact.NamespaceIdentifier;
import ru.yandex.crypta.clients.reactor.artifact.NamespacePermissions;
import ru.yandex.crypta.service.task.model.CryptaTaskStatusChangeInfo;
import ru.yandex.crypta.service.task.model.CryptaTaskStatusChangedEvent;


public class ReactorTaskStatusService implements TaskStatusService {

    private static final Logger LOG = LoggerFactory.getLogger(ReactorTaskStatusService.class);
    private static final String directoryPath = "/crypta/graph/task_statuses";

    private final ReactorClient reactorClient;

    @Inject
    public ReactorTaskStatusService(ReactorClient reactorClient)
    {
        this.reactorClient = reactorClient;

    }


    @Override
    public void reportNewTaskStatus(CryptaTaskStatusChangedEvent changedEvent)
    {
        KeyValueAttributes attributes = new KeyValueAttributes();
        attributes.addArguments(changedEvent.getTaskParameters());
        attributes.addArgument("environment", changedEvent.getEnvironment());
        attributes.addArgument("status", changedEvent.getTaskStatus());

        final String namespacePath = getNamespacePath(changedEvent.getTaskType());
        NamespaceIdentifier artifactIdentifier = new NamespaceIdentifier(namespacePath);

        NamespacePermissions permissions = new NamespacePermissions(1);
        permissions.addRole("crypta", NamespacePermissions.RESPONSIBLE);

        ArtifactCreateRequest artifactCreateRequest = new ArtifactCreateRequest(
                "EVENT",
                artifactIdentifier,
                permissions,
                true,
                true,
                new NamespaceIdentifier("/crypta/Project")
        );

        ArtifactInstantiationResponse instanceResponse = reactorClient.ArtifactInstanceCreate(
                artifactIdentifier,
                "/yandex.reactor.artifact.EventArtifactValueProto",
                null,
                attributes,
                false,
                artifactCreateRequest
        );

        LOG.info("{}: New crypta task in Reactor {}[{}] with status {} [reactorPath = {}]",
                changedEvent.getTaskInstanceId(), changedEvent.getTaskType(), instanceResponse.getArtifactInstanceId(),
                changedEvent.getTaskStatus(), namespacePath
        );
    }


    @Override
    public void reportExistingTaskStatus(String taskInstanceId, String taskType, String taskStatus,
                                         String environment,
                                         Map<String, String> taskParameters)
    {
        final String namespacePath = getNamespacePath(taskType);
        NamespaceIdentifier artifactIdentifier = new NamespaceIdentifier(namespacePath);

        LastArtifactInstanceResponse instanceResponse = reactorClient.GetLastArtifactInstance(artifactIdentifier);

        LOG.info("{}: Existing crypta task {} {} [reactorPath = {}] [instanceId = {}]",
                taskInstanceId, taskType, taskStatus, namespacePath, instanceResponse.getArtifactInstanceId());
    }

    @Override
    public List<CryptaTaskStatusChangeInfo> getTaskInstanceHistory(String taskInstanceId, String taskType, String environment, Map<String, String> taskParameters) {
        return null;
    }

    @Override
    public List<CryptaTaskStatusChangeInfo> getTaskTypeHistory(String taskType, String environment, Map<String, String> taskParameters) {
        return null;
    }

    private String getNamespacePath(String taskType) {
        return String.format("%s/%s", directoryPath, taskType);
    }
}
