package ru.yandex.crypta.service.task;

import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalTime;
import java.time.ZoneOffset;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import javax.inject.Inject;

import org.asynchttpclient.AsyncHttpClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Unit;
import ru.yandex.crypta.lib.proto.EEnvironment;
import ru.yandex.crypta.lib.proto.TSolomonConfig;
import ru.yandex.inside.solomon.SolomonStand;
import ru.yandex.inside.solomon.pusher.SolomonPusher;
import ru.yandex.misc.asyncHttpClient2.AsyncHttpClientUtils;
import ru.yandex.misc.monica.solomon.sensors.PushSensorsData;
import ru.yandex.misc.monica.solomon.sensors.Sensor;

public class SolomonTaskStatusReporter implements TaskStatusReporter {

    private static final Logger LOG = LoggerFactory.getLogger(SolomonTaskStatusReporter.class);

    private static final String PROJECT_CRYPTA = "crypta";
    private static final String CLUSTER_CRYPTA_API = "crypta_api";
    private static final String SERVICE_CRYPTA_TASK_API = "crypta_task_api";

    private static final Map<EEnvironment, SolomonStand> ENV_MAP = Map.of(
            EEnvironment.ENV_PRODUCTION, SolomonStand.PRODUCTION,
            EEnvironment.ENV_TESTING, SolomonStand.PRESTABLE,
            EEnvironment.ENV_DEVELOP, SolomonStand.TESTING
    );

    private final SolomonPusher solomonPusher;

    @Inject
    public SolomonTaskStatusReporter(EEnvironment env, TSolomonConfig solomonConfig) {
        AsyncHttpClient httpClient = AsyncHttpClientUtils.newAsyncHttpClient("SolomonTaskStatusReporter");
        this.solomonPusher = SolomonPusher.ownHttpClient(httpClient)
                .oauthToken(solomonConfig.getToken())
                .pushTo(ENV_MAP.getOrDefault(env, SolomonStand.TESTING))
                .debugNon200()
                .build();
    }

    public SolomonTaskStatusReporter(SolomonPusher solomonPusher) {
        this.solomonPusher = solomonPusher;
    }

    private long calculateDelay(LocalDate dataDate) {
        Instant logReadyTime = dataDate.atTime(LocalTime.MAX).toInstant(ZoneOffset.UTC);
        Duration delay = Duration.between(logReadyTime, Instant.now());

        return delay.getSeconds();
    }

    private void sendTaskDelayToSolomon(String taskType, String taskStatus,
            String environment, Map<String, String> taskParameters)
    {

        if (TaskStatus.SUCCESS.equals(taskStatus) && taskParameters != null && taskParameters.containsKey("date")) {

            Map<String, String> labels = new HashMap<>();
            labels.put("taskType", taskType);
            labels.put("environment", environment);

            LocalDate date = LocalDate.parse(taskParameters.get("date"));
            Sensor sensor = new Sensor(labels, calculateDelay(date));
            sensor = sensor.overrideTs(date.atStartOfDay().toEpochSecond(ZoneOffset.UTC));

            PushSensorsData sensorsData = new PushSensorsData(
                    PROJECT_CRYPTA,
                    CLUSTER_CRYPTA_API,
                    SERVICE_CRYPTA_TASK_API,
                    Collections.emptyMap(),
                    Collections.singletonList(sensor)
            );
            CompletableFuture<Unit> push = solomonPusher.push(sensorsData);
            push.exceptionally(ex -> {
                LOG.error(ex.getMessage());
                return Unit.U;
            });
        }
    }

    @Override
    public void reportTaskStatus(String taskType, String taskStatus,
            String environment, Map<String, String> taskParameters)
    {
        sendTaskDelayToSolomon(taskType, taskStatus, environment, taskParameters);
    }
}
