package ru.yandex.crypta.service.task;

import java.time.Instant;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;

import javax.inject.Inject;

import com.fasterxml.jackson.databind.JsonNode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.crypta.clients.step.StepEventClient;
import ru.yandex.crypta.clients.step.entity.StepEvent;
import ru.yandex.crypta.clients.step.entity.StepEventDescription;
import ru.yandex.crypta.service.task.model.CryptaTaskStatusChangeInfo;
import ru.yandex.crypta.service.task.model.CryptaTaskStatusChangedEvent;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.crypta.clients.utils.JsonUtils.jsonToMap;
import static ru.yandex.crypta.service.task.model.CryptaTaskStatusChangedEvent.ENVIRONMENT_FIELD;
import static ru.yandex.crypta.service.task.model.CryptaTaskStatusChangedEvent.PREDEFINED_FIELDS;
import static ru.yandex.crypta.service.task.model.CryptaTaskStatusChangedEvent.TASK_INSTANCE_ID_FIELD;
import static ru.yandex.crypta.service.task.model.CryptaTaskStatusChangedEvent.TASK_STATUS_FIELD;
import static ru.yandex.crypta.service.task.model.CryptaTaskStatusChangedEvent.TASK_TYPE_FIELD;


public class StepTaskStatusService implements TaskStatusService {

    private static final Logger LOG = LoggerFactory.getLogger(StepTaskStatusService.class);
    private static final String CRYPTA_TASK_EVENT = "cryptaTaskEvent";
    private static final StepEventDescription CRYPTA_TASK_EVENT_TYPE = new StepEventDescription(
            CRYPTA_TASK_EVENT, "Crypta task status changed",
            CryptaTaskStatusChangedEvent.PREDEFINED_FIELDS, Collections.emptyList()
    );
    private final StepEventClient stepEventClient;

    @Inject
    public StepTaskStatusService(StepEventClient stepEventClient)
    {
        this.stepEventClient = stepEventClient;

        Optional<JsonNode> eventDescription = this.stepEventClient.getEventDescription(CRYPTA_TASK_EVENT);
        if (eventDescription.isEmpty()) {
            try {
                this.stepEventClient.registerEventDescription(CRYPTA_TASK_EVENT_TYPE);
            } catch (Exception e) {
                LOG.error("Were not able to register step event type " + CRYPTA_TASK_EVENT_TYPE);
            }
        }
    }


    @Override
    public void reportNewTaskStatus(CryptaTaskStatusChangedEvent changedEvent)
    {
        StepEvent stepEvent = toStepEvent(changedEvent);
        String eventId = stepEventClient.createEvent(stepEvent);

        LOG.info("{}: New crypta task in Step {}[{}] with status {}",
                changedEvent.getTaskInstanceId(), changedEvent.getTaskType(), eventId, changedEvent.getTaskStatus()
        );
    }


    @Override
    public void reportExistingTaskStatus(String taskInstanceId, String taskType, String taskStatus,
            String environment,
            Map<String, String> taskParameters)
    {

        CryptaTaskStatusChangedEvent cryptaTaskStatusChangedEvent = new CryptaTaskStatusChangedEvent(
                taskInstanceId, taskType, taskStatus, environment, taskParameters
        );
        StepEvent stepEvent = toStepEvent(cryptaTaskStatusChangedEvent);
        String eventId = stepEventClient.createEvent(stepEvent);
        LOG.info("{}: Existing crypta task {}[{}] new status {}", eventId, taskType, taskInstanceId, taskStatus);

    }

    @Override
    public List<CryptaTaskStatusChangeInfo> getTaskInstanceHistory(String taskInstanceId, String taskType,
            String environment,
            Map<String, String> taskParameters)
    {
        Map<String, String> params = new HashMap<>(taskParameters);
        params.put(TASK_INSTANCE_ID_FIELD, taskInstanceId);
        params.put(TASK_TYPE_FIELD, taskType);
        params.put(ENVIRONMENT_FIELD, environment);

        List<JsonNode> results = stepEventClient.searchEventInfo(CRYPTA_TASK_EVENT, params, null);
        return results
                .stream()
                .map(this::fromStepFormat)
                .collect(toList());
    }

    @Override
    public List<CryptaTaskStatusChangeInfo> getTaskTypeHistory(String taskType, String environment,
            Map<String, String> taskParameters)
    {
        Map<String, String> params = new HashMap<>(taskParameters);
        params.put(TASK_TYPE_FIELD, taskType);
        params.put(ENVIRONMENT_FIELD, environment);

        List<JsonNode> results = stepEventClient.searchEventInfo(CRYPTA_TASK_EVENT, params, null);
        return results
                .stream()
                .map(this::fromStepFormat)
                .collect(toList());
    }

    private StepEvent toStepEvent(CryptaTaskStatusChangedEvent cryptaTaskStatusChangedEvent) {
        Map<String, String> params = new HashMap<>();
        params.put(TASK_INSTANCE_ID_FIELD, cryptaTaskStatusChangedEvent.getTaskInstanceId());
        params.put(TASK_TYPE_FIELD, cryptaTaskStatusChangedEvent.getTaskType());
        params.put(TASK_STATUS_FIELD, cryptaTaskStatusChangedEvent.getTaskStatus());
        params.put(ENVIRONMENT_FIELD, cryptaTaskStatusChangedEvent.getEnvironment());
        params.putAll(cryptaTaskStatusChangedEvent.getTaskParameters());

        return new StepEvent(CRYPTA_TASK_EVENT, params);
    }


    private CryptaTaskStatusChangeInfo fromStepFormat(JsonNode jsonNode) {
        Map<String, String> params = jsonToMap(jsonNode.get("params"), JsonNode::textValue);

        Map<String, String> customParams = params.entrySet().stream().filter(
                e -> !PREDEFINED_FIELDS.contains(e.getKey())
        ).collect(toMap(Entry::getKey, Entry::getValue));

        CryptaTaskStatusChangedEvent event = new CryptaTaskStatusChangedEvent(
                params.get(TASK_INSTANCE_ID_FIELD),
                params.get(TASK_TYPE_FIELD),
                params.get(TASK_STATUS_FIELD),
                params.get(ENVIRONMENT_FIELD),
                customParams
        );

        String eventId = jsonNode.get("_id").textValue();
        Instant timeCreated = convertToInstant(
                jsonNode.get("time_created").asText()
        );

        return new CryptaTaskStatusChangeInfo(
                eventId,
                timeCreated,
                event
        );

    }

    private Instant convertToInstant(String timeStr) {
        String isoTimeIsoStr = timeStr.replace(" ", "T") + "Z";
        return Instant.parse(isoTimeIsoStr);
    }


}
