package ru.yandex.crypta.service.tx;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ForkJoinPool;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.inject.Inject;

import NCrypta.Transaction.TTransaction;
import NCrypta.TransactionSource.ETransactionSource;
import yabs.proto.Profile.TSourceUniq;
import yabs.proto.UserProfile;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.crypta.clients.bigb.BigbClient;
import ru.yandex.crypta.clients.bigb.BigbIdType;
import ru.yandex.crypta.lib.proto.identifiers.EIdType;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.crypta.proto.PublicTransaction;
import ru.yandex.inside.yt.kosher.cypress.RangeLimit;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;

public class DefaultTxService implements TxService {
    private final BigbClient bigbClient;
    private final YtService ytService;
    private final ForkJoinPool customForkJoin;

    private static final long SECONDS_IN_YEAR = 31536000L;
    private static final Map<String, String> CATEGORY_TYPE_TO_NAME = Map.of(
            "eda", "Еда",
            "edadeal", "Едадил",
            "kinopoisk", "Кинопоиск",
            "lavka", "Лавка",
            "market", "Маркет",
            "taxi", "Такси",
            "trust", "Другие покупки в Яндексе",
            "zapravki", "Заправки"
    );

    @Inject
    public DefaultTxService(BigbClient bigbClient, YtService ytService) {
        this.bigbClient = bigbClient;
        this.ytService = ytService;
        this.customForkJoin = new ForkJoinPool();
    }

    @Override
    public List<TTransaction> getTransactions(ETransactionSource txSource, BigbIdType bigbIdType, String bigbIdValue) {
        var uniqs = getUniqs(bigbIdType, bigbIdValue);

        return runInCustomForkJoin(() ->
                getTransactionsStream(txSource, uniqs)
                        .sorted(Comparator.comparing(TTransaction::getTimestamp).reversed())
                        .collect(Collectors.toList())
        );
    }

    @Override
    public List<TTransaction> getAllTransactions(BigbIdType bigbIdType, String bigbIdValue) {
        List<TSourceUniq> uniqs;
        if (bigbIdType == BigbIdType.CRYPTA_ID) {
            uniqs = Collections.singletonList(TSourceUniq.newBuilder()
                    .setUserId(bigbIdValue)
                    .setIdType(TSourceUniq.EIdType.CRYPTA_ID2)
                    .build()
            );
        } else {
            uniqs = getUniqs(bigbIdType, bigbIdValue);
        }

        return runInCustomForkJoin(() ->
                List.of(ETransactionSource.values()).parallelStream()
                        .flatMap(txSource -> getTransactionsStream(txSource, uniqs))
                        .sorted(Comparator.comparing(TTransaction::getTimestamp).reversed())
                        .collect(Collectors.toList())
        );
    }

    @Override
    //this method returns transactions that are not older than 365 days
    public List<PublicTransaction> getPublicTransactions(BigbIdType bigbIdType, String bigbIdValue) {

        var allTransactions = getAllTransactions(bigbIdType, bigbIdValue);

        Map<String, Float> categoryToPrice = new HashMap<>();
        List<PublicTransaction> publicTransactions = new ArrayList<>();

        long startOfThePeriodTimestamp = Instant.now().getEpochSecond() - SECONDS_IN_YEAR;

        allTransactions.forEach(
            transaction -> {
                var key = transaction.getSource();
                if (key.equals("eda") && transaction.getSeller().equals("Яндекс.Лавка")) {
                    key = transaction.getSeller();
                }

                if (transaction.getTimestamp() < startOfThePeriodTimestamp || !CATEGORY_TYPE_TO_NAME.containsKey(key) || transaction.getStatus().equals("Cancelled")) {
                    return;
                }
                var value = categoryToPrice.containsKey(key) ? categoryToPrice.get(key) : 0;
                categoryToPrice.put(key, value + transaction.getItemUnitPriceRub() * transaction.getItemQuantity());
            }
        );

        categoryToPrice.forEach((key, value) -> {
            var transaction = PublicTransaction.newBuilder()
                    .setCategory(CATEGORY_TYPE_TO_NAME.getOrDefault(key, key))
                    .setPrice(value)
                    .build();
            publicTransactions.add(transaction);
            }
        );
        return publicTransactions;
    }

    private Stream<TTransaction> getTransactionsStreamByCryptaId(ETransactionSource txSource, String cryptaId) {
        List<TTransaction> transactions = new ArrayList<>();

        // Can be rewritten using getHahnRpcAsync in non-blocking manner with Futures
        var path = TxPaths.getTxTablePath(txSource);
        if (ytService.getHahn().cypress().exists(path)) {
            ytService.getHahn().tables().read(
                    path.withExact(
                            new RangeLimit(Cf.list(YTree.stringNode(cryptaId)), -1, -1)
                    ),
                    YTableEntryTypes.nativeProto(TTransaction.class, 1, true),
                    (Consumer<TTransaction>) transactions::add
            );
        }
        return transactions.stream();
    }

    private Stream<TTransaction> getTransactionsStream(ETransactionSource txSource, List<TSourceUniq> uniqs) {
        return uniqs.parallelStream()
                .filter(uniq -> EIdType.CRYPTA_ID == getIdType(uniq.getIdType()))
                .flatMap(uniq -> getTransactionsStreamByCryptaId(txSource, uniq.getUserId()));
    }

    private List<TSourceUniq> getUniqs(BigbIdType bigbIdType, String bigIdValue) {
        return bigbClient.getBigbDataProto(bigbIdType, bigIdValue).getGluedUniqsList();
    }

    private static EIdType getIdType(TSourceUniq.EIdType bigbIdType) {
        return bigbIdType.getValueDescriptor().getOptions().getExtension(UserProfile.cryptaEnumId);
    }

    /**
     * Run in custom thread pool, don't block common
     */
    private <T> T runInCustomForkJoin(Callable<T> job) {
        try {
            return customForkJoin.submit(job).get();
        } catch (InterruptedException | ExecutionException e) {
            throw new RuntimeException(e);
        }
    }
}
