package ru.yandex.crypta.service.useragents;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import javax.inject.Inject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.lib.proto.EEnvironment;
import ru.yandex.crypta.lib.proto.TYqlConfig;
import ru.yandex.crypta.lib.schedulers.Schedulers;
import ru.yandex.crypta.lib.yt.PathUtils;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.transactions.utils.YtTransactionsUtils;
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeListNodeImpl;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.transactions.Transaction;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.yql.YqlDataSource;
import ru.yandex.yql.settings.YqlProperties;


public class JooqUseragentsService implements UseragentsService {

    private final static Logger LOG = LoggerFactory.getLogger(JooqUseragentsService.class);
    private static final String JDBC_YQL_HAHN_CONNETCTION_STRING = "jdbc:yql://yql.yandex.net:443/hahn?syntaxVersion=1";
    private static final String FIELD_DETAILEDDEVICETYPE = "detaileddevicetype";
    private static final String FIELD_USERAGENT = "useragent";
    private static final boolean DO_NOT_PING_ANCESTOR_TRANSACTIONS = false;
    private static final YPath hitLogYPath = YPath.simple("//logs/bs-hit-log/1d");
    private static final String useragentsModTimeAttribute = "_mod_time";
    private final YtService yt;
    private final TYqlConfig yqlConfig;
    private final ScheduledExecutorService executor;
    private final String topUaDeviceTypesTable;
    private final YPath topUaDeviceTypesYPath;

    @Inject
    public JooqUseragentsService(
            YtService yt,
            Schedulers schedulers,
            TYqlConfig yqlConfig,
            EEnvironment environment
    )
    {
        this.yqlConfig = yqlConfig;
        this.yt = yt;
        this.executor = schedulers.getExecutor();
        this.topUaDeviceTypesTable =
                String.format("home/crypta/%s/portal/ads/Useragents", PathUtils.toPath(environment));
        this.topUaDeviceTypesYPath = YPath.simple(String.format("//%s", topUaDeviceTypesTable));
    }

    private String getLatestHitLogTableName() {
        List<String> hitLogsLatestTableList = yt.getHahn().cypress()
                .get(hitLogYPath)
                .asMap()
                .keySet()
                .stream()
                .sorted()
                .collect(Collectors.toList());

        hitLogsLatestTableList = hitLogsLatestTableList
                .subList(hitLogsLatestTableList.size() - 20, hitLogsLatestTableList.size());

        List<String> latest = hitLogsLatestTableList.stream()
                .filter(node -> ((YTreeListNodeImpl) yt.getHahn().cypress()
                        .get(hitLogYPath.child(node).attribute("locks"))).size() == 0)
                .collect(Collectors.toList());

        return latest.get(latest.size() - 1);
    }

    private long getUATableModTime() {
        return yt.getHahn().cypress().get(topUaDeviceTypesYPath.attribute(useragentsModTimeAttribute)).longValue();
    }

    private void setUATableModTime(Transaction transaction) {
        Option<GUID> transactionId = Option.of(transaction.getId());
        YPath attributePath = topUaDeviceTypesYPath.attribute(useragentsModTimeAttribute);
        long attributeValue = System.currentTimeMillis();

        yt.getHahn().cypress().set(transactionId.getOrNull(), DO_NOT_PING_ANCESTOR_TRANSACTIONS, attributePath, attributeValue);
    }

    private boolean uATableisOlderThan(int value, TimeUnit timeUnit) {
        long now = System.currentTimeMillis();
        long modTime = getUATableModTime();

        return (now - modTime) > timeUnit.toMillis(value);
    }

    @Override
    @SuppressWarnings("FutureReturnValueIgnored")
    public Map<String, List<String>> getTopUseragents() {
        ListF<YTreeMapNode> deviceUseragents = Cf.arrayList();

        if (!yt.getHahn().cypress().exists(topUaDeviceTypesYPath)) {
            executor.schedule(this::updateTopUseragents, 1, TimeUnit.MILLISECONDS);
            throw Exceptions.unavailable();
        }

        yt.getHahn().tables().read(topUaDeviceTypesYPath,
                YTableEntryTypes.YSON, (Consumer<YTreeMapNode>) deviceUseragents::add);

        if (uATableisOlderThan(1, TimeUnit.DAYS)) {
            executor.schedule(this::updateTopUseragents, 1, TimeUnit.MILLISECONDS);
        }

        return deviceUseragents
                .stream()
                .collect(Collectors.groupingBy(rec -> rec.getString(FIELD_DETAILEDDEVICETYPE),
                        Collectors.mapping(
                                rec -> rec.getString(FIELD_USERAGENT),
                                Collectors.toList()
                        ))
                );
    }

    private void withYtTransaction(Consumer<Transaction> body) {
        try {
            YtTransactionsUtils
                    .withTransaction(yt.getHahn(), Duration.ofSeconds(10), Optional.of(Duration.ofSeconds(1)),
                            transaction -> {
                                body.accept(transaction);
                                return true;
                            });
        } catch (RuntimeException e) {
            throw Exceptions.illegal(e.getMessage());
        }
    }

    @Override
    public boolean updateTopUseragents() {
        LOG.info("Top devicetype useragents stalled, updating.");

        withYtTransaction(transaction -> {
            YqlDataSource yqlDataSource = new YqlDataSource(
                    JDBC_YQL_HAHN_CONNETCTION_STRING,
                    new YqlProperties().withCredentials(yqlConfig.getUser(), yqlConfig.getToken())
            );
            String query = String.format(
                    "PRAGMA yt.ExternalTx=\"%6$s\";\n" +

                            "$ranks = (SELECT %1$s,\n" +
                            "         %2$s,\n" +
                            "         %5$s,\n" +
                            "         row_number() over w as rk\n" +
                            "    FROM (\n" +
                            "        SELECT %1$s,\n" +
                            "               %2$s,\n" +
                            "               count(*) as %5$s\n" +
                            "        FROM `%3$s` as dt\n" +
                            "        GROUP BY %1$s, %2$s\n" +
                            "    ) as t\n" +
                            "    WINDOW w as (partition by %1$s order by %5$s desc)\n" +
                            ");\n" +

                            "INSERT INTO `%4$s`\n" +
                            "WITH TRUNCATE\n" +

                            "SELECT %1$s, %2$s, %5$s FROM $ranks WHERE rk < 11\n" +
                            "ORDER BY %5$s DESC\n" +
                            "LIMIT 100;",
                    FIELD_DETAILEDDEVICETYPE, FIELD_USERAGENT,
                    hitLogYPath.child(getLatestHitLogTableName()).toString().substring(2),
                    topUaDeviceTypesTable, "cnt", transaction.getId().toString());

            try (Connection connection = yqlDataSource.getConnection();
                 Statement statement = connection.createStatement())
            {
                statement.execute(query);
            } catch (SQLException e) {
                LOG.error("Failed to execute query", e);
            }

            setUATableModTime(transaction);
        });
        return true;
    }
}
