package ru.yandex.crypta.common.ws.auth;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;

import javax.annotation.Priority;
import javax.inject.Inject;
import javax.ws.rs.HttpMethod;
import javax.ws.rs.Priorities;
import javax.ws.rs.container.ContainerRequestContext;
import javax.ws.rs.container.ContainerRequestFilter;
import javax.ws.rs.container.ResourceInfo;

import org.apache.http.client.utils.URIBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

import ru.yandex.crypta.clients.blackbox.AuthResult;
import ru.yandex.crypta.clients.blackbox.BlackboxClient;
import ru.yandex.crypta.clients.idm.RoleService;
import ru.yandex.crypta.clients.tvm.TvmClient;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.common.ws.Security;
import ru.yandex.crypta.common.ws.jersey.ContextIdentifiers;
import ru.yandex.crypta.common.ws.jersey.IpAware;
import ru.yandex.crypta.common.ws.jersey.exception.OptionsRequestedException;
import ru.yandex.crypta.common.ws.solomon.Solomon;
import ru.yandex.crypta.lib.proto.TBlackboxConfig;
import ru.yandex.crypta.lib.proto.TDevelopmentFlags;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.passport.tvmauth.CheckedServiceTicket;

@Priority(Priorities.AUTHENTICATION)
public class AuthRequestFilter extends IpAware implements ContainerRequestFilter {

    public static final String X_CRYPTA_LOGIN = "X-Crypta-Login";
    public static final String X_YA_SERVICE_TICKET = "X-Ya-Service-Ticket";

    private static final Logger LOG = LoggerFactory.getLogger(AuthRequestFilter.class);
    private static final String YANDEXUID_COOKIE = "yandexuid";
    private static final String SESSION_ID_COOKIE = "Session_id";
    private static final String AUTHORIZATION_HEADER = "Authorization";
    private static final String HOST_HEADER = "Host";
    private static final String ORIGIN_HEADER = "Origin";

    private static final Rate UNAUTHENTICATED_RATE = Solomon.REGISTRY.rate("request.unauthenticated");
    private static final Rate TVM_RATE = Solomon.REGISTRY.rate("request.auth.tvm");
    private static final Rate SESSIONID_RATE = Solomon.REGISTRY.rate("request.auth.sessionid");
    private static final Rate OAUTH_RATE = Solomon.REGISTRY.rate("request.auth.oauth");

    private static final String YANDEX_TEAM_DOMAIN_REGEX = "^.+\\.yandex-team\\.ru$";

    private final BlackboxClient blackbox;
    private final RoleService roleService;
    private final TDevelopmentFlags flags;
    private final TvmClient tvmClient;
    private final TBlackboxConfig blackboxConfig;

    @Inject
    private ResourceInfo resourceInfo;

    @Inject
    public AuthRequestFilter(
            BlackboxClient blackbox, RoleService roleService, TDevelopmentFlags flags, TvmClient tvmClient,
            TBlackboxConfig blackboxConfig
    ) {
        this.blackbox = blackbox;
        this.blackboxConfig = blackboxConfig;
        this.roleService = roleService;
        this.flags = flags;
        this.tvmClient = tvmClient;
    }

    @SafeVarargs
    private static <A, T> Function<A, Optional<T>> chain(Function<A, Optional<T>>... functions) {
        return (arg) -> {
            for (var function : functions) {
                var optionalResult = function.apply(arg);
                if (optionalResult.isPresent()) {
                    return optionalResult;
                }
            }
            return Optional.empty();
        };
    }

    @Override
    public void filter(ContainerRequestContext requestContext) {
        if (isOptions(requestContext)) {
            throw new OptionsRequestedException();
        }

        if (isAuthNotRequired()) {
            return;
        }

        var yandexuid = requestContext.getCookies().get(YANDEXUID_COOKIE);
        if (Objects.nonNull(yandexuid)) {
            MDC.put("yandexuid", yandexuid.getValue());
        }

        if (isAuthDisabled()) {
            setUpSecurity(requestContext, Authenticator.develop(flags.getDebugUserName()));
            return;
        }

        var optionalAuthenticator = chain(this::tryOauth, this::trySessionid, this::tryTvm).apply(requestContext);

        if (optionalAuthenticator.isPresent()) {
            var authenticator = optionalAuthenticator.get();
            setUpSecurity(requestContext, authenticator);
            MDC.put("auth_method", authenticator.getMethod().name());
            MDC.put("login", authenticator.getLogin());
            MDC.put("puid", authenticator.getPuid());
        } else {
            handleUnauthenticated(requestContext);
        }
    }

    private boolean isOptions(ContainerRequestContext requestContext) {
        return requestContext.getMethod().equals(HttpMethod.OPTIONS);
    }

    private boolean isAuthDisabled() {
        return flags.getDisableAuth();
    }

    private boolean isAuthNotRequired() {
        return Objects.nonNull(resourceInfo.getResourceMethod().getAnnotation(AuthNotRequired.class));
    }

    private void handleUnauthenticated(ContainerRequestContext requestContext) {
        var headers = requestContext.getHeaders();
        var requestUrl = requestContext.getUriInfo().getRequestUri();
        var remoteAddr = getRemoteAddr();
        var method = requestContext.getMethod();
        MDC.clear();
        MDC.put("request_url", requestUrl.toASCIIString());
        MDC.put("request_from", remoteAddr);
        MDC.put("request_headers", Security.screened(headers).toString());
        MDC.put("request_method", method);
        UNAUTHENTICATED_RATE.inc();
        throw Exceptions.notAuthenticated("Failed to authenticate");
    }

    private void setUpSecurity(ContainerRequestContext requestContext, Authenticator authenticator) {
        requestContext.setSecurityContext(new IdmSecurityContext(flags, roleService, authenticator));
        ContextIdentifiers.storeAuthInfo(requestContext, authenticator);
    }

    private String buildPassportRedirectLocation(ContainerRequestContext requestContext, String path) {
        String scheme = "https";
        String schemeWithSlashes = scheme + "://";

        String retpathUrl = schemeWithSlashes + requestContext.getHeaderString("Host");
        String requestReferrer = requestContext.getHeaderString("Referer");
        if (!Objects.isNull(requestReferrer)) {
            retpathUrl = requestReferrer;
        }

        try {
            URIBuilder uriBuilder = new URIBuilder()
                    .setScheme(scheme)
                    .setHost(blackboxConfig.getPassportUrl())
                    .setPath(path)
                    .setParameter("retpath", retpathUrl);

            return uriBuilder.build().toString();
        } catch (URISyntaxException e) {
            return schemeWithSlashes + blackboxConfig.getPassportUrl() + path;
        }
    }

    private String buildUpdateAuthenticationUrl(ContainerRequestContext requestContext) {
        return buildPassportRedirectLocation(requestContext, "/auth/update/");
    }

    private String buildObtainAuthenticationUrl(ContainerRequestContext requestContext) {
        return buildPassportRedirectLocation(requestContext, "/auth");
    }

    private Optional<Authenticator> checkSuccess(AuthResult authResult, AuthMethod method, ContainerRequestContext requestContext) {

        if (Objects.isNull(authResult)) {
            return Optional.empty();
        }
        if (!authResult.isOk()) {
            if (authResult.getMethod().equals(AuthResult.Method.SESSIONID)) {
                if (authResult.needReset()) {
                    throw Exceptions.outdatedAuthentication(buildUpdateAuthenticationUrl(requestContext));
                }

                throw Exceptions.badAuthentication(buildObtainAuthenticationUrl(requestContext));
            }

            return Optional.empty();
        }
        LOG.debug("Authenticated {} by {}", authResult.getLogin(), authResult.getMethod());
        return Optional.of(new Authenticator(authResult.getLogin(), authResult.getPuid(), method));
    }

    private Optional<Authenticator> tryOauth(ContainerRequestContext requestContext) {
        var authorizationHeaders = requestContext.getHeaders().get(AUTHORIZATION_HEADER);

        if (Objects.isNull(authorizationHeaders)) {
            return Optional.empty();
        }
        if (authorizationHeaders.size() != 1) {
            return Optional.empty();
        }
        AuthResult result = blackbox.oauth(authorizationHeaders.get(0), getTypedRemoteAddr());
        var authenticator = checkSuccess(result, AuthMethod.OAUTH, requestContext);
        authenticator.ifPresent(x -> OAUTH_RATE.inc());
        return authenticator;
    }

    private Optional<Authenticator> trySessionid(ContainerRequestContext requestContext) {
        var sessionidCookie = requestContext.getCookies().get(SESSION_ID_COOKIE);
        var origin = requestContext.getHeaders().getFirst(ORIGIN_HEADER);
        var hosts = requestContext.getHeaders().get(HOST_HEADER);

        if (hosts.size() != 1) {
            return Optional.empty();
        }

        if (Objects.isNull(sessionidCookie)) {
            if (!Objects.isNull(origin) && Objects.equals(URI.create(origin).getHost(), blackboxConfig.getPassportUrl())) {
                return Optional.empty();
            }

            if (requestContext.getUriInfo().getRequestUri().getHost().matches(YANDEX_TEAM_DOMAIN_REGEX)) {
                throw Exceptions.badAuthentication(buildObtainAuthenticationUrl(requestContext));
            }

            return Optional.empty();
        }

        var result = blackbox.sessionid(sessionidCookie.getValue(), getTypedRemoteAddr(), hosts.get(0));
        var authenticator = checkSuccess(result, AuthMethod.SESSION_ID, requestContext);
        authenticator.ifPresent(x -> SESSIONID_RATE.inc());
        return authenticator;
    }

    private Optional<Authenticator> tryTvm(ContainerRequestContext requestContext) {
        var serviceTicketBody = requestContext.getHeaders().getFirst(X_YA_SERVICE_TICKET);

        if (Objects.isNull(serviceTicketBody)) {
            LOG.debug("Authenticated by tvm failed, service ticket is null");
            return Optional.empty();
        }

        CheckedServiceTicket serviceTicket = tvmClient.checkServiceTicket(serviceTicketBody);

        if (!serviceTicket.booleanValue()) {
            LOG.debug("Authenticated by tvm failed, {}", getResult(serviceTicket));
            return Optional.empty();
        }

        var login = Integer.toString(serviceTicket.getSrc());

        LOG.debug("Authenticated {} by {}", login, "TVM");
        TVM_RATE.inc();
        return Optional.of(new Authenticator(login, "", AuthMethod.TVM));
    }

    private static String getResult(CheckedServiceTicket result) {
        return String.format("result: status= %s, debugInfo= %s, src= %s, issuerUid= %s",
                result.getStatus(), result.debugInfo(), result.getSrc(), result.getIssuerUid());
    }

}
