package ru.yandex.crypta.common.ws.jersey;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.time.Duration;
import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.util.Objects;
import java.util.UUID;

import javax.ws.rs.container.ContainerRequestContext;
import javax.ws.rs.container.ContainerResponseContext;
import javax.ws.rs.container.ResourceInfo;
import javax.ws.rs.core.MultivaluedMap;

import org.slf4j.MDC;

import ru.yandex.crypta.common.ws.auth.Authenticator;
import ru.yandex.crypta.common.ws.solomon.Solomon;
import ru.yandex.library.svnversion.VcsVersion;
import ru.yandex.monlib.metrics.histogram.HistogramCollector;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.Rate;

public class ContextIdentifiers {

    public static final String X_CRYPTA_REQUEST_ID = "X-Crypta-Request-ID";
    public static final String X_CRYPTA_INSTANCE_ID = "X-Crypta-Instance-ID";
    public static final String X_CRYPTA_HOST = "X-Crypta-Host";
    public static final String X_CRYPTA_VERSION = "X-Crypta-Version";
    public static final String X_CRYPTA_PROCESSING_TIME = "X-Crypta-Processing-Time";
    public static final String X_CRYPTA_REQUEST_TIME = "X-Crypta-Request-Time";

    public static final String X_CRYPTA_LOGIN = "X-Crypta-Login";
    public static final String X_CRYPTA_PUID = "X-Crypta-Puid";

    // fields matching Deploy log table schema
    public static final String REQUEST_ID = "request_id";
    public static final String USER_ID = "user_id";

    private static final DateTimeFormatter TIME_FORMATTER = DateTimeFormatter.ISO_INSTANT;
    private static final VcsVersion VCS = new VcsVersion(ContextIdentifiers.class);
    private static final HistogramCollector PROCESSING_TIME_BINS = Histograms.exponential(10, 2, 10);
    private static final Histogram PROCESSING_TIME_HISTOGRAM =
            Solomon.REGISTRY.histogramRate("request.processing_time", PROCESSING_TIME_BINS);
    private static final Rate RESPONSE_RATE = Solomon.REGISTRY.rate("request.response.total");

    @SuppressWarnings({"StaticVariableName"})
    public static String INSTANCE_ID;

    @SuppressWarnings({"StaticVariableName"})
    public static String HOSTNAME;

    @SuppressWarnings({"StaticVariableName"})
    public static String VERSION;

    static {
        INSTANCE_ID = "instance-" + UUID.randomUUID().toString();
        try {
            HOSTNAME = InetAddress.getLocalHost().getHostName();
        } catch (UnknownHostException e) {
            HOSTNAME = "unknown";
        }
        try {
            VERSION = String.valueOf(VCS.getProgramSvnRevision());
        } catch (Throwable e) {
            VERSION = "unknown";
        }
    }

    private ContextIdentifiers() {
    }

    private static String createRequestId() {
        return "request-" + UUID.randomUUID().toString();
    }

    private static String getOrCreateRequestId(MultivaluedMap<String, String> headers) {
        if (headers.containsKey(X_CRYPTA_REQUEST_ID)) {
            return headers.getFirst(X_CRYPTA_REQUEST_ID);
        } else {
            return createRequestId();
        }
    }

    public static void prepareRequest(ContainerRequestContext context,
            MultivaluedMap<String, String> headers) {
        String requestId = getOrCreateRequestId(headers);

        context.setProperty(X_CRYPTA_REQUEST_ID, requestId);
        context.setProperty(X_CRYPTA_INSTANCE_ID, INSTANCE_ID);
        context.setProperty(X_CRYPTA_HOST, HOSTNAME);
        context.setProperty(X_CRYPTA_VERSION, VERSION);
        context.setProperty(X_CRYPTA_REQUEST_TIME, Instant.now());

        MDC.put(X_CRYPTA_REQUEST_ID, requestId);
        MDC.put(REQUEST_ID, requestId);
    }

    public static void finalizeResponse(ContainerRequestContext request, ContainerResponseContext response,
            ResourceInfo resourceInfo)
    {
        response.getHeaders().putSingle(X_CRYPTA_REQUEST_ID, request.getProperty(X_CRYPTA_REQUEST_ID));
        response.getHeaders().putSingle(X_CRYPTA_INSTANCE_ID, request.getProperty(X_CRYPTA_INSTANCE_ID));
        response.getHeaders().putSingle(X_CRYPTA_HOST, request.getProperty(X_CRYPTA_HOST));
        response.getHeaders().putSingle(X_CRYPTA_VERSION, request.getProperty(X_CRYPTA_VERSION));

        Instant requestTime = (Instant) request.getProperty(X_CRYPTA_REQUEST_TIME);
        response.getHeaders().putSingle(X_CRYPTA_REQUEST_TIME, TIME_FORMATTER.format(requestTime));
        long processingTime = Duration.between(requestTime, Instant.now()).toMillis();

        recordProcessingTimeHistogram(processingTime);
        recordProcessingTimeHistogramForResource(resourceInfo, processingTime);
        incRateForStatus(response.getStatus());
        incRateForStatusAndResource(resourceInfo, response.getStatus());
        incRateTotal();

        response.getHeaders().putSingle(X_CRYPTA_PROCESSING_TIME, processingTime + "ms");
    }

    public static void storeAuthInfo(ContainerRequestContext context, Authenticator authenticator) {
        context.setProperty(X_CRYPTA_LOGIN, authenticator.getLogin());
        context.setProperty(X_CRYPTA_PUID, authenticator.getPuid());
        MDC.put(USER_ID, authenticator.getLogin());
        MDC.put(X_CRYPTA_LOGIN, authenticator.getLogin());
    }

    public static String getPuid(ContainerRequestContext context) {
        return (String) context.getProperty(X_CRYPTA_PUID);
    }

    public static String getRequestId(ContainerRequestContext context) {
        return (String) context.getProperty(X_CRYPTA_REQUEST_ID);
    }

    private static void recordProcessingTimeHistogram(long processingTime) {
        PROCESSING_TIME_HISTOGRAM.record(processingTime);
    }

    private static void recordProcessingTimeHistogramForResource(ResourceInfo resourceInfo, long processingTime) {
        if (nonEmptyResourceInfo(resourceInfo)) {
            Labels labels = getLabelsForResource(resourceInfo);
            Solomon.REGISTRY
                    .histogramRate("request.processing_time", labels, PROCESSING_TIME_BINS)
                    .record(processingTime);
        }
    }

    private static boolean nonEmptyResourceInfo(ResourceInfo resourceInfo) {
        return Objects.nonNull(resourceInfo) &&
                Objects.nonNull(resourceInfo.getResourceMethod()) &&
                Objects.nonNull(resourceInfo.getResourceClass());
    }

    private static Labels getLabelsForResource(ResourceInfo resourceInfo) {
        try {
            return Labels.of(
                    "method", resourceInfo.getResourceMethod().getName(),
                    "class", resourceInfo.getResourceClass().getName()
            );
        } catch (Throwable e) {
            return Labels.of(
                    "method",
                    "unknown",
                    "class",
                    "unknown"
            );
        }
    }

    private static void incRateForStatusAndResource(ResourceInfo resourceInfo, int status) {
        if (nonEmptyResourceInfo(resourceInfo)) {
            Labels labels = getLabelsForResource(resourceInfo);
            Solomon.REGISTRY.rate("request.response." + status, labels).inc();
        }
    }

    private static void incRateForStatus(int status) {
        Solomon.REGISTRY.rate("request.response." + status).inc();
    }

    private static void incRateTotal() {
        RESPONSE_RATE.inc();
    }

}
