package ru.yandex.crypta.common.ws.jersey;

import javax.ws.rs.container.ContainerRequestFilter;
import javax.ws.rs.container.ContainerResponseFilter;
import javax.ws.rs.ext.ContextResolver;
import javax.ws.rs.ext.ExceptionMapper;

import org.glassfish.jersey.internal.inject.AbstractBinder;
import org.glassfish.jersey.internal.inject.InjectionManager;
import org.glassfish.jersey.media.multipart.MultiPartFeature;
import org.glassfish.jersey.message.GZipEncoder;
import org.glassfish.jersey.server.ResourceConfig;
import org.glassfish.jersey.server.ServerProperties;
import org.glassfish.jersey.server.filter.EncodingFilter;
import org.glassfish.jersey.server.filter.RolesAllowedDynamicFeature;
import org.glassfish.jersey.server.spi.Container;
import org.glassfish.jersey.server.spi.ContainerLifecycleListener;
import org.reflections.Reflections;

import ru.yandex.crypta.common.ws.auth.AuthRequestFilter;
import ru.yandex.crypta.common.ws.jersey.exception.ExceptionsRoot;
import ru.yandex.crypta.common.ws.jersey.filter.CorsResponseFilter;
import ru.yandex.crypta.common.ws.jersey.filter.LoggingRequestFilter;
import ru.yandex.crypta.common.ws.jersey.filter.LoggingResponseFilter;
import ru.yandex.crypta.common.ws.jersey.filter.ThrottlingRequestFilter;
import ru.yandex.crypta.common.ws.jersey.resolver.CustomJacksonObjectMapperContextResolver;
import ru.yandex.crypta.common.ws.jersey.resource.HealthResource;
import ru.yandex.crypta.common.ws.jersey.resource.SolomonResource;
import ru.yandex.crypta.common.ws.jersey.validation.CustomValidationConfigContextResolver;
import ru.yandex.crypta.common.ws.swagger.SwaggerResource;
import ru.yandex.crypta.common.ws.swagger.SwaggerStaticResource;

public class ReflectionJerseyResourceConfig extends ResourceConfig {

    private final Class<?> restRoot;
    private final AbstractBinder binder;
    private InjectionManager injectionManager;

    public ReflectionJerseyResourceConfig(Class<?> restRoot, AbstractBinder binder) {
        this.restRoot = restRoot;
        this.binder = binder;
        this.injectionManager = null;
        baseInit();
        init();
    }

    private void baseInit() {
        register(binder);
        Package restPackage = restRoot.getPackage();
        packages(restPackage.getName());
        registerApplicationComponents(getReflections(restPackage));
        registerCustomValidation();
        registerCustomSerialization();
        registerSwaggerResources();
        registerFilters();
        registerMultipartSupport();
        registerAllDefaultExceptionMappers();
        registerDefaultResources();
        registerSecurity();
        registerLocatorExtractor();
        initProperties();
        setUpContentEncoding();
    }

    private void setUpContentEncoding() {
        EncodingFilter.enableFor(this, GZipEncoder.class);
    }

    private void initProperties() {
        property(ServerProperties.WADL_FEATURE_DISABLE, true);
    }

    private void registerLocatorExtractor() {
        register(new ContainerLifecycleListener() {
            @Override
            public void onStartup(Container container) {
                injectionManager = container.getApplicationHandler().getInjectionManager();
            }

            @Override
            public void onReload(Container container) {

            }

            @Override
            public void onShutdown(Container container) {

            }
        });
    }

    private void registerSecurity() {
        register(RolesRestrictedDynamicFeature.class);
        register(RolesAllowedDynamicFeature.class);
        register(AuthRequestFilter.class);
    }

    private void registerDefaultResources() {
        register(HealthResource.class);
        register(SolomonResource.class);
    }

    private void registerCustomSerialization() {
        register(CustomJacksonObjectMapperContextResolver.class);
    }

    private void registerCustomValidation() {
        register(CustomValidationConfigContextResolver.class);
    }

    private void registerApplicationComponents(Reflections reflections) {
        registerAll(reflections, ExceptionMapper.class);
        registerAll(reflections, ContainerRequestFilter.class);
        registerAll(reflections, ContainerResponseFilter.class);
        registerAll(reflections, ContextResolver.class);
    }

    private void registerAllDefaultExceptionMappers() {
        registerAll(getReflections(ExceptionsRoot.class.getPackage()), ExceptionMapper.class);
    }

    private void registerMultipartSupport() {
        register(MultiPartFeature.class);
    }

    private void registerFilters() {
        register(LoggingRequestFilter.class);
        register(LoggingResponseFilter.class);
        register(CorsResponseFilter.class);
        register(ThrottlingRequestFilter.class);
    }

    private void registerSwaggerResources() {
        register(SwaggerResource.class);
        register(SwaggerStaticResource.class);
    }

    protected void init() {
    }

    private <T> void registerAll(Reflections reflections, Class<T> theClass) {
        reflections.getSubTypesOf(theClass).forEach(this::register);
    }

    private Reflections getReflections(Package thePackage) {
        return new Reflections(thePackage.getName());
    }

    public AbstractBinder getBinder() {
        return binder;
    }

    public InjectionManager getInjectionManager() {
        return injectionManager;
    }

}
