package ru.yandex.crypta.common.ws.jersey;

import java.io.IOException;
import java.util.Arrays;

import javax.annotation.Priority;
import javax.annotation.security.RolesAllowed;
import javax.ws.rs.ForbiddenException;
import javax.ws.rs.Priorities;
import javax.ws.rs.container.ContainerRequestContext;
import javax.ws.rs.container.ContainerRequestFilter;
import javax.ws.rs.container.DynamicFeature;
import javax.ws.rs.container.ResourceInfo;
import javax.ws.rs.core.FeatureContext;
import javax.ws.rs.core.SecurityContext;

import org.glassfish.jersey.server.internal.LocalizationMessages;
import org.glassfish.jersey.server.model.AnnotatedMethod;

import ru.yandex.crypta.idm.Roles;


public class RolesRestrictedDynamicFeature implements DynamicFeature  {
    private void registerConfig(final FeatureContext configuration, RolesForbidden rolesForbidden, RolesAllowed rolesAllowed) {
        String[] rolesAllowedValue = rolesAllowed != null ? rolesAllowed.value() : null;
        configuration.register(new RolesRestrictedRequestFilter(rolesForbidden.value(), rolesAllowedValue));
    }

    @Override
    public void configure(final ResourceInfo resourceInfo, final FeatureContext configuration) {
        final var annotatedMethod = new AnnotatedMethod(resourceInfo.getResourceMethod());

        // RolesForbidden on the method takes precedence over RolesForbidden on the class
        var rolesForbidden = annotatedMethod.getAnnotation(RolesForbidden.class);
        var rolesAllowed = annotatedMethod.getAnnotation(RolesAllowed.class);
        if (rolesForbidden != null) {
            registerConfig(configuration, rolesForbidden, rolesAllowed);
            return;
        }

        rolesForbidden = resourceInfo.getResourceClass().getAnnotation(RolesForbidden.class);
        rolesAllowed = resourceInfo.getResourceClass().getAnnotation(RolesAllowed.class);
        if (rolesForbidden != null) {
            registerConfig(configuration, rolesForbidden, rolesAllowed);
        }
    }

    @Priority(Priorities.AUTHORIZATION) // authorization filter - should go after any authentication filters
    private static class RolesRestrictedRequestFilter implements ContainerRequestFilter {
        private final String[] rolesForbidden;
        private final String[] rolesAllowed;

        RolesRestrictedRequestFilter(final String[] rolesForbidden, final String[] rolesAllowed) {
            this.rolesForbidden = (rolesForbidden != null) ? rolesForbidden : new String[] {};
            this.rolesAllowed = (rolesAllowed != null) ? rolesAllowed : new String[] {};
        }

        @Override
        public void filter(final ContainerRequestContext requestContext) throws IOException {
            var securityContext = requestContext.getSecurityContext();

            for (final String forbiddenRole : rolesForbidden) {
                if (
                        !isAuthenticated(requestContext) || (
                            securityContext.isUserInRole(forbiddenRole)
                            && Arrays.stream(rolesAllowed).noneMatch(securityContext::isUserInRole)
                            && !isAdmin(securityContext)
                        )
                ) {
                    throw new ForbiddenException(LocalizationMessages.USER_NOT_AUTHORIZED());
                }
            }
        }

        private static boolean isAdmin(SecurityContext securityContext) {
            return securityContext.isUserInRole(Roles.Lab.ADMIN) || securityContext.isUserInRole(Roles.ADMIN);
        }

        private static boolean isAuthenticated(final ContainerRequestContext requestContext) {
            return requestContext.getSecurityContext().getUserPrincipal() != null;
        }
    }
}
