package ru.yandex.crypta.common.ws.json.protobuf;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.PropertyNamingStrategy.PropertyNamingStrategyBase;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.ser.std.StdSerializer;
import com.google.protobuf.ByteString;
import com.google.protobuf.Descriptors.EnumValueDescriptor;
import com.google.protobuf.Descriptors.FieldDescriptor;
import com.google.protobuf.MapEntry;
import com.google.protobuf.MessageOrBuilder;
import com.google.protobuf.util.JsonFormat;

import ru.yandex.crypta.lib.proto.TWsIoConfig;

import static java.lang.String.format;

public class ProtobufSerializer extends StdSerializer<MessageOrBuilder> {

    private final Map<Class<?>, JsonSerializer<Object>> serializerCache;
    private final JsonFormat.Printer jsonFormat;
    private final TWsIoConfig config;

    public ProtobufSerializer(TWsIoConfig config) {
        super(MessageOrBuilder.class);

        this.serializerCache = new ConcurrentHashMap<>();
        this.jsonFormat = JsonFormat.printer().sortingMapKeys().includingDefaultValueFields();
        this.config = config;
    }

    private static boolean writeEnumsUsingIndex(SerializerProvider config) {
        return config.isEnabled(SerializationFeature.WRITE_ENUMS_USING_INDEX);
    }

    private static IOException unrecognizedType(JsonGenerator generator, FieldDescriptor field) throws IOException {
        String error = format("Unrecognized java type '%s' for field %s", field.getJavaType(), field.getFullName());
        throw new JsonMappingException(generator, error);
    }

    private void serializeJsonFormat(MessageOrBuilder messageOrBuilder, JsonGenerator generator)
            throws IOException
    {
        generator.writeRawValue(jsonFormat.print(messageOrBuilder));
    }

    private void serializeDefault(MessageOrBuilder message, JsonGenerator generator,
            SerializerProvider serializerProvider)
            throws IOException
    {
        generator.writeStartObject();

        PropertyNamingStrategyBase namingStrategy =
                new PropertyNamingStrategyWrapper(serializerProvider.getConfig().getPropertyNamingStrategy());

        for (FieldDescriptor field : message.getDescriptorForType().getFields()) {
            if (field.isRepeated()) {
                if (field.isMapField()) {
                    List<?> valueMap = (List<?>) message.getField(field);

                    generator.writeObjectFieldStart(namingStrategy.translate(field.getName()));

                    for (Object mapEntry : valueMap) {
                        MapEntry<?, ?> entry = (MapEntry) mapEntry;
                        generator.writeFieldName(entry.getKey().toString());
                        writeValue(field, entry.getValue(), generator, serializerProvider);
                    }

                    generator.writeEndObject();
                } else {
                    List<?> valueList = (List<?>) message.getField(field);

                    generator.writeArrayFieldStart(namingStrategy.translate(field.getName()));
                    for (Object subValue : valueList) {
                        writeValue(field, subValue, generator, serializerProvider);
                    }
                    generator.writeEndArray();
                }
            } else if (field.getContainingOneof() != null) {
                FieldDescriptor oneofFieldDescriptor = message.getOneofFieldDescriptor(field.getContainingOneof());

                if (field.equals(oneofFieldDescriptor) && message.hasField(oneofFieldDescriptor)) {
                    generator.writeFieldName(namingStrategy.translate(oneofFieldDescriptor.getName()));
                    writeValue(oneofFieldDescriptor, message.getField(oneofFieldDescriptor), generator,
                            serializerProvider);
                }

            } else {
                generator.writeFieldName(namingStrategy.translate(field.getName()));
                writeValue(field, message.getField(field), generator, serializerProvider);

            }
        }

        generator.writeEndObject();
    }

    @Override
    public void serialize(MessageOrBuilder message, JsonGenerator generator, SerializerProvider serializerProvider)
            throws IOException
    {
        if (config.getEnableJsonFormatSerialization()) {
            serializeJsonFormat(message, generator);
        } else {
            serializeDefault(message, generator, serializerProvider);
        }
    }

    private void writeValue(FieldDescriptor field, Object value, JsonGenerator generator,
            SerializerProvider serializerProvider) throws IOException
    {
        switch (field.getJavaType()) {
            case INT:
                generator.writeNumber((Integer) value);
                break;
            case LONG:
                generator.writeNumber((Long) value);
                break;
            case FLOAT:
                generator.writeNumber((Float) value);
                break;
            case DOUBLE:
                generator.writeNumber((Double) value);
                break;
            case BOOLEAN:
                generator.writeBoolean((Boolean) value);
                break;
            case STRING:
                generator.writeString((String) value);
                break;
            case ENUM:
                EnumValueDescriptor enumDescriptor = (EnumValueDescriptor) value;

                if (writeEnumsUsingIndex(serializerProvider)) {
                    generator.writeNumber(enumDescriptor.getNumber());
                } else {
                    generator.writeString(enumDescriptor.getName());
                }
                break;
            case BYTE_STRING:
                generator.writeString(
                        serializerProvider.getConfig().getBase64Variant().encode(((ByteString) value).toByteArray()));
                break;
            case MESSAGE:
                Class<?> subType = value.getClass();

                JsonSerializer<Object> serializer = serializerCache.get(subType);
                if (serializer == null) {
                    serializer = serializerProvider.findValueSerializer(value.getClass(), null);
                    serializerCache.put(subType, serializer);
                }

                serializer.serialize(value, generator, serializerProvider);
                break;
            default:
                throw unrecognizedType(generator, field);
        }
    }
}
