package ru.yandex.crypta.common.ws.solomon;

import java.lang.management.ManagementFactory;
import java.lang.management.ThreadInfo;
import java.lang.management.ThreadMXBean;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

public class JvmThreads {

    public static final Pattern THREAD_NAME_ENDS_WITH_NUMBER_REGEX = Pattern.compile("(.*?)(-?)(\\d+)$");

    /**
     * Group threads by provided grouping function and add metrics per group
     * Note that only current state of threads is taken into account
     *
     * @param groupBy grouping function
     */
    public static void addThreadGroupMetrics(MetricRegistry registry, String metricName,
                                             Function<ThreadInfo, String> groupBy) {
        ThreadMXBean threadMXBean = ManagementFactory.getThreadMXBean();

        Map<String, List<ThreadInfo>> byGroupingName = Stream
                .of(threadMXBean.getThreadInfo(threadMXBean.getAllThreadIds()))
                .collect(Collectors.groupingBy(
                        groupBy,
                        Collectors.toList()
                ));

        for (String groupingName : byGroupingName.keySet()) {
            MetricRegistry groupingMetrics = registry.subRegistry(metricName, groupingName);
            for (Thread.State state : Thread.State.values()) {
                List<ThreadInfo> threads = byGroupingName.get(groupingName);

                Labels stateLabel = Labels.of("state", state.name());
                groupingMetrics.lazyGaugeInt64("jvm.threads.count", stateLabel,
                        () -> threads.stream()
                                .filter(threadInfo -> threadInfo != null && threadInfo.getThreadState() == state)
                                .count()
                );
            }
        }

    }

    /**
     * Heuristically extract pool name from common thread sequence naming in thread pools like
     * - pool1, pool2
     * - pool-1-1, pool-1-2
     */
    public static String extractPoolNameByLastNumber(ThreadInfo threadInfo) {
        Matcher matcher = THREAD_NAME_ENDS_WITH_NUMBER_REGEX.matcher(threadInfo.getThreadName());
        if (matcher.matches()) {
            String heuristicPoolName = matcher.group(1);
            if (!heuristicPoolName.isEmpty()) {
                return heuristicPoolName;
            }
        }

        return threadInfo.getThreadName();

    }


}
