package ru.yandex.crypta.common.ws.swagger;

import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Objects;

import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriBuilder;

import io.swagger.annotations.Api;

import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.swagger.SwaggerUiResource;

@Path("/")
@Api(hidden = true)
@Produces("text/html")
public class SwaggerStaticResource {

    public static final String CSP_POLICY = "default-src 'self'; script-src 'self' 'unsafe-inline'; style-src 'self' 'unsafe-inline'";
    public static final String CSP_HEADER = "Content-Security-Policy";

    @GET
    @Path("/")
    public Response getIndex() {
        return goStatic("index.html");
    }

    @GET
    @Path("swagger")
    public Response getSwagger() {
        return goStatic("index.html");
    }

    @Path("favicon.ico")
    @GET
    public Response getIcon() {
        return goStatic( "favicon-32x32.ico");
    }

    @Path("swagger/{filename}")
    @GET
    public Response getStatic(@PathParam("filename") String filename) {
        try (InputStream is = getResourceAsStream(filename)) {
            if (Objects.isNull(is)) {
                throw Exceptions.notFound();
            }
            byte[] contents = is.readAllBytes();
            String tag = Integer.toHexString(Arrays.hashCode(contents));
            String contentType = determineContentType(filename);
            return Response.ok()
                    .entity(contents)
                    .header(CSP_HEADER, CSP_POLICY)
                    .type(contentType)
                    .tag(tag)
                    .build();
        } catch (IOException e) {
            throw Exceptions.internal("Failed to read file");
        }
    }

    private InputStream getResourceAsStream(String filename) {
        ClassLoader classLoader = SwaggerUiResource.getClassLoader();
        return classLoader.getResourceAsStream(SwaggerUiResource.getResourcePath() + filename);
    }

    private Response goStatic(String filename) {
        UriBuilder builder = UriBuilder.fromPath("");
        builder.path("swagger");
        builder.path(filename);
        return Response.seeOther(builder.build()).build();
    }

    private String determineContentType(String filename) {
        if (filename.endsWith(".html")) {
            return MediaType.TEXT_HTML;
        }
        if (filename.endsWith(".css")) {
            return "text/css";
        }
        if (filename.endsWith(".png")) {
            return "image/png";
        }
        if (filename.endsWith(".js")) {
            return "application/javascript";
        }
        return MediaType.TEXT_PLAIN;
    }

}
