package ru.yandex.crypta.common.ws.swagger;

import java.io.File;
import java.lang.annotation.Annotation;
import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.databind.BeanDescription;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.type.SimpleType;
import com.google.common.collect.ImmutableList;
import com.google.protobuf.ByteString;
import io.swagger.converter.ModelConverter;
import io.swagger.converter.ModelConverterContext;
import io.swagger.converter.ModelConverters;
import io.swagger.jackson.ModelResolver;
import io.swagger.jaxrs.config.BeanConfig;
import io.swagger.models.Model;
import io.swagger.models.ModelImpl;
import io.swagger.models.Operation;
import io.swagger.models.Path;
import io.swagger.models.Response;
import io.swagger.models.Swagger;
import io.swagger.models.properties.FileProperty;
import io.swagger.models.properties.Property;
import io.swagger.models.properties.RefProperty;
import io.swagger.models.properties.StringProperty;
import io.swagger.util.Json;
import io.swagger.util.PrimitiveType;

import ru.yandex.crypta.common.ws.jersey.ContextIdentifiers;
import ru.yandex.crypta.common.ws.jersey.resource.HealthResource;
import ru.yandex.crypta.common.ws.json.protobuf.ProtobufModule;
import ru.yandex.crypta.common.ws.swagger.proto.ProtobufModuleWithIntrospection;

public class SwaggerUtils {

    public static final ImmutableList<HttpError> BASIC_HTTP_ERRORS = ImmutableList.of(
            HttpError.of(429, "Too many requests. Simple retry would suffice."),
            HttpError.of(403, "User not authenticated or not authorized to access this resource."),
            HttpError.of(405, "Method not allowed. The request method (GET/..) is not supported. " +
                    "Should not happen when API is generated."),
            HttpError.of(400, "Invalid request. Likely caused by the application itself."),
            HttpError.of(404, "Not found. Likely caused by the application itself."),
            HttpError.of(406, "Not acceptable. The requested content type is not supported. " +
                    "Should not happen when API is generated."),
            HttpError.of(500, "Server error. Major issue that should be reported."),
            HttpError.of(503, "Service unavailable. Major issue that should be reported."),
            HttpError.of(504, "Backend timeout. Likely caused by the application platform.")
    );

    static {
        {
            ProtobufModule module = new ProtobufModuleWithIntrospection();
            Json.mapper().registerModule(module);
        }
    }

    private SwaggerUtils() {

    }

    /**
     * Creates swagger for provided package.
     *
     * @param thePackage package to scan resources from
     * @return swagger instance
     */
    public static Swagger createFromPackage(Package thePackage, String prefix) {
        String packageName = thePackage.getName();
        BeanConfig beanConfig = new BeanConfig();
        beanConfig.setScan(true);

        hackToUseFullyQualifiedNames();

        beanConfig.setResourcePackage(packageName);
        beanConfig.scanAndRead();
        beanConfig.setResourcePackage(HealthResource.class.getPackage().getName());
        beanConfig.scanAndRead();

        Swagger swagger = beanConfig.getSwagger();
        swagger.setBasePath("/" + prefix);
        swagger.setSchemes(null);

        return swagger;
    }

    private static void hackToUseFullyQualifiedNames() {
        ModelConverters.getInstance().addConverter(new ModelResolver(Json.mapper()) {

            private String normalize(String name) {
                return name.replace('$', '.');
            }

            @Override
            protected String _findTypeName(JavaType type, BeanDescription beanDesc) {
                if (type instanceof SimpleType) {
                    if (isFile((SimpleType) type)) {
                        return super._findTypeName(type, beanDesc);
                    }
                    return normalize(type.toCanonical());
                } else {
                    return super._findTypeName(type, beanDesc);
                }
            }

            @Override
            public Property resolveProperty(Type type,
                    ModelConverterContext context,
                    Annotation[] annotations,
                    Iterator<ModelConverter> next)
            {
                if (type instanceof SimpleType) {
                    if (isByteString((SimpleType) type)) {
                        return PrimitiveType.createProperty("String");
                    }
                }
                return super.resolveProperty(type, context, annotations, next);
            }

            @Override
            public Model resolve(Type type, ModelConverterContext context, Iterator<ModelConverter> next) {

                return super.resolve(type, context, next);
            }

            private boolean isByteString(SimpleType type) {
                return type.getRawClass().equals(ByteString.class);
            }

            private boolean isFile(SimpleType type) {
                return type.getRawClass().equals(File.class);
            }
        });
    }

    /**
     * Hacks swagger to be able to have proper file responses.
     * <p>
     * Apparently there was no file response support for a while. It was introduced quite
     * recently but still got no good integration with swagger-annotation.
     * <p>
     * The method dives into swagger definition and finds responses with
     * schemas referencing the File definition. These references are
     * replaced with file schemas.
     *
     * @param swagger swagger to patch
     */
    public static void setupProperFileResponseSchema(Swagger swagger) {
        final String fileDefinitionReference = "#/definitions/File";
        Map<String, Path> paths = swagger.getPaths();
        if (paths == null) {
            return;
        }
        paths.values().forEach(
                (path) -> {
                    path.getOperations().forEach((operation) -> {
                        operation.getResponses().values().forEach((response) -> {
                            Property schema = response.getSchema();
                            if (schema instanceof RefProperty) {
                                RefProperty refSchema = (RefProperty) schema;
                                if (refSchema.get$ref().equals(fileDefinitionReference)) {
                                    FileProperty replacementSchema = new FileProperty();
                                    response.setSchema(replacementSchema);
                                }
                            }
                        });
                    });
                }
        );
    }

    /**
     * Hacks swagger to have proper model for jackson's JsonNode.
     *
     * JsonNode is basically a union/map-like type so it should not have any
     * properties in the model.
     *
     * @param swagger swagger to patch
     */
    public static void makeJsonNodeArbitrary(Swagger swagger) {
        final String jsonNodeDefinitionReference = JsonNode.class.getName();
        Map<String, Model> definitions = swagger.getDefinitions();
        if (definitions == null) {
            return;
        }
        definitions.put(jsonNodeDefinitionReference, new ModelImpl());
    }

    public static void fillDefaultOperationResponses(Swagger swagger, Class errorResponseClass,
            List<HttpError> httpErrors)
    {
        Property responseSchema = ModelConverters.getInstance().readAsProperty(errorResponseClass);
        Map<String, Path> paths = swagger.getPaths();
        if (paths == null) {
            return;
        }

        for (Path path : paths.values()) {
            for (Operation operation : path.getOperations()) {
                Map<String, Response> responses = operation.getResponses();
                httpErrors.forEach((httpError) -> {
                    Response httpErrorSchema = new Response()
                            .description(httpError.getDescription())
                            .schema(responseSchema);
                    responses.put(String.valueOf(httpError.getCode()), httpErrorSchema);
                });
            }
        }

        Map<String, Model> responseDefinitions = ModelConverters.getInstance().read(errorResponseClass);
        Map<String, Model> definitions = swagger.getDefinitions();
        if (definitions == null) {
            swagger.setDefinitions(new HashMap<>(responseDefinitions));
        } else {
            definitions.putAll(responseDefinitions);
        }

        setupPossibleResponseHeaders(swagger);
    }

    private static void setupPossibleResponseHeaders(Swagger swagger) {
        Property requestIdProperty = new StringProperty().description("Request ID to be reported");
        Property instanceIdProperty = new StringProperty().description("Instance ID to be reported");
        Property hostProperty = new StringProperty().description("Host to be reported");
        swagger.getPaths()
                .values()
                .stream()
                .map(Path::getOperations)
                .flatMap(List::stream)
                .map(Operation::getResponses)
                .flatMap(responses -> responses.values().stream())
                .forEach(response -> {
                    response.addHeader(ContextIdentifiers.X_CRYPTA_REQUEST_ID, requestIdProperty);
                    response.addHeader(ContextIdentifiers.X_CRYPTA_INSTANCE_ID, instanceIdProperty);
                    response.addHeader(ContextIdentifiers.X_CRYPTA_HOST, hostProperty);
                });
    }

    public static class HttpError {

        private final int code;
        private final String description;

        public HttpError(int code, String description) {
            this.code = code;
            this.description = description;
        }

        public static HttpError of(int code, String description) {
            return new HttpError(code, description);
        }

        public int getCode() {
            return code;
        }

        public String getDescription() {
            return description;
        }
    }

}
