package ru.yandex.crypta.lib.yt;

import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.inject.Inject;

import io.netty.channel.nio.NioEventLoopGroup;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.crypta.common.ws.solomon.Solomon;
import ru.yandex.crypta.lib.proto.TYtConfig;
import ru.yandex.inside.yt.kosher.impl.YtConfiguration;
import ru.yandex.inside.yt.kosher.impl.YtUtils;
import ru.yandex.misc.io.IoUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.monlib.metrics.JvmThreads;
import ru.yandex.yt.ytclient.bus.DefaultBusConnector;
import ru.yandex.yt.ytclient.proxy.YtClient;
import ru.yandex.yt.ytclient.proxy.YtCluster;
import ru.yandex.yt.ytclient.rpc.RpcCredentials;

public class DefaultYtService implements YtService {

    private static final Logger LOG = LoggerFactory.getLogger(DefaultYtService.class);

    private final YtClient hahnRpcClient;
    private final ru.yandex.inside.yt.kosher.CloseableYt hahnClient;
    private final ru.yandex.inside.yt.kosher.async.CloseableYt hahnAsyncClient;
    private final YtClient senecaRpcClient;
    private final ExecutorService ytHeavyExecutor;

    @Inject
    public DefaultYtService(TYtConfig config) {
        // hahn
        YtConfiguration hahnConfig = YtConfiguration.builder()
                .withApiHost(config.getProxy())
                .withUser(config.getRpcUser())
                .withToken(config.getToken())
                .build();
        hahnClient = YtUtils.http(hahnConfig);

        ytHeavyExecutor = Executors.newCachedThreadPool();
        JvmThreads.addExecutorMetrics("YtHeavyExecutor", ytHeavyExecutor, Solomon.REGISTRY);

        LOG.info("YT Hahn RPC client: init");
        NioEventLoopGroup hahnEventLoop = new NioEventLoopGroup();
        JvmThreads.addExecutorMetrics("YtHahnEventLoop", hahnEventLoop, Solomon.REGISTRY);

        hahnRpcClient = YtClient.builder()
                .setCluster(config.getProxy())
                .setRpcCredentials(
                        new RpcCredentials(config.getRpcUser(), config.getToken())
                ).setOwnBusConnector(
                        new DefaultBusConnector(hahnEventLoop)
                )
                .setHeavyExecutor(ytHeavyExecutor)
                .build();

        LOG.info("YT Hahn RPC async client: init");
        hahnAsyncClient = ru.yandex.inside.yt.kosher.async.Yt.builder(hahnConfig)
                .rpc()
                .withYtClient(hahnRpcClient)
                .build();

        LOG.info("YT Hahn RPC client: waiting for proxies");
        hahnRpcClient.waitProxies().join();
        LOG.info("YT Hahn RPC client: init done");

        // seneca
        Validate.notEmpty(config.getSenecaHosts());
        LOG.info("YT Seneca RPC client: init for " + config.getSenecaHosts());
        List<YtCluster> senecaClusters = Stream
                .of(config.getSenecaHosts().split(","))
                .map(YtCluster::new)
                .collect(Collectors.toList());

        NioEventLoopGroup senecaEventLoop = new NioEventLoopGroup();
        JvmThreads.addExecutorMetrics("YtSenecaEventLoop", senecaEventLoop, Solomon.REGISTRY);

        senecaRpcClient = YtClient.builder()
                .setClusters(senecaClusters)
                .setRpcCredentials(
                        new RpcCredentials(config.getRpcUser(), config.getToken())
                ).setOwnBusConnector(
                        new DefaultBusConnector(senecaEventLoop)
                )
                .setHeavyExecutor(ytHeavyExecutor)
                .build();

        LOG.info("YT Seneca RPC client: waiting for proxies");
        senecaRpcClient.waitProxies().join();
        LOG.info("YT Seneca RPC client: init done");

    }

    @Override
    public ru.yandex.inside.yt.kosher.Yt getHahn() {
        return hahnClient;
    }

    @Override
    public YtClient getHahnRpc() {
        return hahnRpcClient;
    }

    @Override
    public ru.yandex.inside.yt.kosher.async.Yt getHahnAsync() {
        return hahnAsyncClient;
    }

    @Override
    public YtClient getSenecaRpc() {
        return senecaRpcClient;
    }

    public void close() {
        IoUtils.closeQuietly(hahnClient);
        IoUtils.closeQuietly(hahnRpcClient);
        IoUtils.closeQuietly(hahnAsyncClient);
        IoUtils.closeQuietly(senecaRpcClient);
        ytHeavyExecutor.shutdown();
    }
}
