package ru.yandex.crypta.lib.yt;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Consumer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.RangeLimit;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.tables.YTableEntryType;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.misc.lang.number.UnsignedLong;
import ru.yandex.yt.ytclient.proxy.LookupRowsRequest;
import ru.yandex.yt.ytclient.proxy.YtClient;
import ru.yandex.yt.ytclient.tables.TableSchema;

public class YtReadingUtils {

    private static final Logger LOG = LoggerFactory.getLogger(YtReadingUtils.class);

    public static RangeLimit exact(ListF<YTreeNode> keys) {
        return new RangeLimit(keys, -1, -1);
    }

    public static RangeLimit exact(YTreeNode key) {
        return exact(Cf.arrayList(key));
    }

    public static RangeLimit exact(String key) {
        return exact(YTree.stringNode(key));
    }

    public static RangeLimit exact(UnsignedLong key) {
        return exact(YTree.unsignedIntegerNode(key.longValue()));
    }

    public static RangeLimit exact(List<String> multiKey) {
        ListF<YTreeNode> keys = Cf.wrap(multiKey).map(YTree::stringNode);
        return new RangeLimit(keys, -1, -1);
    }

    public static <T> Optional<T> lookupSingleRow(Yt yt, YPath path, YTreeMapNode key, YTableEntryType<T> entryType) {
        List<T> results = new ArrayList<>();
        yt.tables().lookupRows(path,
                YTableEntryTypes.YSON,
                Cf.list(key),
                entryType,
                (Consumer<T>) results::add);
        if (results.isEmpty()) {
            return Optional.empty();
        }
        if (results.size() != 1) {
            throw Exceptions.illegal("Got multiple rows while one requested");
        }
        return Optional.of(results.get(0));
    }

    public static List<YTreeMapNode> lookupRows(YtClient ytClient, YPath path, TableSchema schema, List<?> key,
                                                long timeoutMsec) {
        LookupRowsRequest request = new LookupRowsRequest(path.toString(), schema.toLookup())
                .addFilter(key)
                // preventive request timeout to discover where timeout really occurs
                // TODO: remove in prod
                .setTimeout(Duration.ofMillis(timeoutMsec - 200));

        Instant reqStart = Instant.now();
        try {
            return ytClient.lookupRows(request)
                    .get(timeoutMsec, TimeUnit.MILLISECONDS)
                    .getYTreeRows();
        } catch (InterruptedException | ExecutionException e) {
            throw new RuntimeException("Failed to lookup rows", e);
        } catch (TimeoutException e) {
            dumpForkJoinPool();

            Instant reqEnd = Instant.now();
            Duration timeElapsed = Duration.between(reqStart, reqEnd);

            throw new RuntimeException("Failed to lookup rows, took " + timeElapsed, e);
        } finally {
            Instant reqEnd = Instant.now();
            Duration timeElapsed = Duration.between(reqStart, reqEnd);
            LOG.info("Lookup rows took " + timeElapsed);
        }
    }

    private static void dumpForkJoinPool() {
        StringBuilder sb = new StringBuilder();
        sb.append("Threads stats\n");
        sb.append("ForkJoinPool parallelism: ").append(ForkJoinPool.commonPool().getParallelism());
        sb.append('\n');
        sb.append("ForkJoinPool pool size: ").append(ForkJoinPool.commonPool().getPoolSize());
        sb.append('\n');

        for (Map.Entry<Thread, StackTraceElement[]> threadEntry : Thread.getAllStackTraces().entrySet()) {
            Thread thread = threadEntry.getKey();
            StackTraceElement[] stackTrace = threadEntry.getValue();

            sb.append("==== ");
            sb.append(thread.getName()).append(" ").append(thread.getState());
            sb.append(" at :\n");
            for (StackTraceElement stackTraceElement : stackTrace) {
                sb.append(stackTraceElement).append('\n');
            }
            sb.append('\n');

        }

        LOG.warn(sb.toString());
    }
}
