#include "parse_pb_options.h"

#include <util/system/env.h>
#include <util/string/builder.h>

#include <google/protobuf/descriptor.pb.h>
#include <google/protobuf/dynamic_message.h>
#include <crypta/lib/native/proto_serializer/proto_serializer.h>
#include <crypta/lib/proto/extensions/extensions.pb.h>
#include <library/cpp/getoptpb/proto/confoption.pb.h>
#include <library/cpp/protobuf/dynamic_prototype/generate_file_descriptor_set.h>
#include <library/cpp/protobuf/dynamic_prototype/dynamic_prototype.h>

#include <util/generic/set.h>
#include <util/generic/queue.h>

namespace {
    const google::protobuf::EnumValueDescriptor* FindEnumValue(const google::protobuf::EnumDescriptor* descriptor, const TString& value) {
        if (!value) {
            return nullptr;
        }

        TString ciValue = to_lower(value);
        for (int i = 0; i < descriptor->value_count(); ++i) {
            auto valueDescriptor = descriptor->value(i);
            TString plainValue = to_lower(valueDescriptor->name());
            if (ciValue == plainValue) {
                return valueDescriptor;
            }
            TString valValue = to_lower(valueDescriptor->options().GetExtension(NGetoptPb::Val));
            if (ciValue == valValue) {
                return valueDescriptor;
            }
        }
        return nullptr;
    }

    bool IsIntegerType(NProtoBuf::FieldDescriptor::CppType field_type) {
        using T = ::google::protobuf::FieldDescriptor::CppType;

        switch(field_type) {
            case T::CPPTYPE_INT32:
            case T::CPPTYPE_UINT32:
            case T::CPPTYPE_INT64:
            case T::CPPTYPE_UINT64: {
                return true;
            }
            default:
                return false;
        }
    }

    constexpr const char* const DEVELOP = "develop";
    constexpr const char* const TESTING = "testing";
    constexpr const char* const STABLE = "stable";
    constexpr const char* const PRODUCTION = "production";

    TMaybe<TString> GetValueFromEnvExtensions(const NProtoBuf::FieldDescriptor& fieldDescr) {
        TMaybe<TString> value;

        if (fieldDescr.options().HasExtension(NCryptaOpts::SelectFromEnv)) {
            TString currentEnv = GetEnv(fieldDescr.options().GetExtension(NCryptaOpts::SelectFromEnv), DEVELOP);
            currentEnv.to_lower();

            if (currentEnv == DEVELOP) {
                value = fieldDescr.options().GetExtension(NCryptaOpts::Develop);
            } else if (currentEnv == TESTING) {
                value = fieldDescr.options().GetExtension(NCryptaOpts::Testing);
            } else if (currentEnv == STABLE || currentEnv == PRODUCTION) {
                value = fieldDescr.options().GetExtension(NCryptaOpts::Production);
            }
        }

        if (fieldDescr.options().HasExtension(NCryptaOpts::FromEnv)) {
            const auto& envValue = GetEnv(fieldDescr.options().GetExtension(NCryptaOpts::FromEnv));
            if (!envValue.empty()) {
                value = envValue;
            }
        }

        return value;
    }
}

namespace NCrypta {
    using namespace NProtoBuf;

    void ParsePbOptions(int argc, const char** argv, google::protobuf::Message& options) {
        TString error;
        Y_ENSURE(NGetoptPb::GetoptPb(argc, argv, options, error, {.DumpConfig = false}), "Can not parse command line options: " << error);
    }

    void ParsePbOptionsExtended(int argc, const char** argv, google::protobuf::Message& message) {
        auto descr = message.GetDescriptor();
        auto fds = GenerateFileDescriptorSet(descr);

        auto result = ParsePbOptionsDynamic(argc, argv, fds, descr->name());
        NProtoSerializer::FromString(message, result);
    }

    bool ProcessCryptaExtensions(NProtoBuf::Message* message) {
        using T = ::google::protobuf::FieldDescriptor::CppType;
        auto descr = message->GetDescriptor();
        auto refl = message->GetReflection();

        const ::google::protobuf::OneofDescriptor* prevOneofDescriptor = nullptr;
        bool oneofPartialySet = false;

        bool made_a_change = false;
        for (int i = 0; i < descr->field_count(); ++i) {
            auto field_descr = descr->field(i);
            auto oneof_descr = field_descr->real_containing_oneof();
            auto field_type = field_descr->cpp_type();

            if (prevOneofDescriptor != oneof_descr || (oneof_descr == nullptr && prevOneofDescriptor == nullptr)) {
                prevOneofDescriptor = oneof_descr;
                oneofPartialySet = false;
            }

            if ((field_type == T::CPPTYPE_MESSAGE) && !field_descr->is_repeated()) {
                ::google::protobuf::DynamicMessageFactory factory;
                auto sub_message = factory.GetPrototype(field_descr->message_type())->New();
                bool sub_message_made_a_change = ProcessCryptaExtensions(sub_message);

                Y_ENSURE(!(oneofPartialySet && sub_message_made_a_change), "Two different fields in oneof are partially set by environment variables");

                if (sub_message_made_a_change) {
                    if (oneof_descr != nullptr) {
                        oneofPartialySet = true;
                    }
                    refl->MutableMessage(message, field_descr)->MergeFrom(*sub_message);
                }

                made_a_change |= sub_message_made_a_change;
            } else if (field_type == T::CPPTYPE_ENUM && !field_descr->is_repeated())  {
                if (field_descr->options().HasExtension(NCryptaOpts::FromEnv)) {
                    TString value = GetEnv(field_descr->options().GetExtension(NCryptaOpts::FromEnv), "");
                    if (!value.empty()) {
                        auto enumDescriptor = field_descr->enum_type();
                        auto enumValue = FindEnumValue(enumDescriptor, value);
                        Y_ENSURE(enumValue, "Invalid value for enum: " << value);
                        refl->SetEnum(message, field_descr, enumValue);
                        made_a_change = true;
                    }
                }
            } else if (field_type == T::CPPTYPE_STRING && !field_descr->is_repeated()) {
                const auto& value = GetValueFromEnvExtensions(*field_descr);

                if (value.Defined()) {
                    refl->SetString(message, field_descr, *value);
                    made_a_change = true;
                }
            } else if (IsIntegerType(field_type) && !field_descr->is_repeated()) {
                const auto& value = GetValueFromEnvExtensions(*field_descr);

                if (value.Defined()) {
                    made_a_change = true;
                    switch (field_type) {
                        case T::CPPTYPE_INT32:
                            refl->SetInt32(message, field_descr, FromString<i32>(*value));
                            break;
                        case T::CPPTYPE_UINT32:
                            refl->SetUInt32(message, field_descr, FromString<ui32>(*value));
                            break;
                        case T::CPPTYPE_INT64:
                            refl->SetInt64(message, field_descr, FromString<i64>(*value));
                            break;
                        case T::CPPTYPE_UINT64:
                            refl->SetUInt64(message, field_descr, FromString<ui64>(*value));
                            break;
                        default:
                            made_a_change = false;
                            break;
                    }
                }

            }
        }

        return made_a_change;
    }

    TString ParsePbOptionsDynamic(int argc, const char** argv, const NProtoBuf::FileDescriptorSet& fds, const TString& message_type) {
        auto prototype = TDynamicPrototype::Create(fds, message_type);

        auto environ_message = prototype->Create();
        ProcessCryptaExtensions(environ_message.Get());
        // parse message from options and config file
        auto message = prototype->Create();
        NCrypta::ParsePbOptions(argc, argv, *message);
        // merge environ with conf
        environ_message->MergeFrom(*message);

        return NProtoSerializer::ToString(*environ_message);
    }

    TString ParsePbOptionsDynamic(int argc, const char** argv, const TString& serialized_fds, const TString& message_type) {
        return ParsePbOptionsDynamic(argc, argv, NProtoSerializer::CreateFromString<FileDescriptorSet>(serialized_fds), message_type);
    }
}
