#include <google/protobuf/text_format.h>
#include <library/cpp/testing/unittest/registar.h>
#include <util/system/env.h>
#include <util/system/tempfile.h>
#include <util/stream/file.h>

#include <crypta/lib/native/cmd_args/parse_pb_options.h>
#include <crypta/lib/native/cmd_args/ut/cmd_args_ut.pb.h>

Y_UNIT_TEST_SUITE(CryptaPbOptsExtensions) {
    int argc = 1;
    const char* argv[] = {"./a.out"};

    Y_UNIT_TEST(SelectFromEnv) {
        SetEnv("TEST1_VAR", "testing");
        SetEnv("TEST2_VAR", "production");
        SetEnv("TEST3_VAR", "develop");

        auto config = NCrypta::ParsePbOptionsExtended<TTestMessage>(argc, argv);
        UNIT_ASSERT_EQUAL("Testing", config.GetTestString1());
        UNIT_ASSERT_EQUAL("Production", config.GetTestString2());
        UNIT_ASSERT_EQUAL("Develop", config.GetTestString3());
    }
    Y_UNIT_TEST(SelectFromEnvDefault) {
        auto config = NCrypta::ParsePbOptionsExtended<TTestMessage>(argc, argv);
        UNIT_ASSERT_EQUAL("Develop", config.GetTestString4());
    }
    Y_UNIT_TEST(FromEnv) {
        SetEnv("TEST5_VAR", "TestString5Value");

        auto config = NCrypta::ParsePbOptionsExtended<TTestMessage>(argc, argv);
        UNIT_ASSERT_EQUAL("TestString5Value", config.GetTestString5());
        UNIT_ASSERT_EQUAL("", config.GetTestString6());
    }
    Y_UNIT_TEST(FromEnvInt) {
        SetEnv("TEST7_VAR", "100");
        auto config = NCrypta::ParsePbOptionsExtended<TTestMessage>(argc, argv);

        UNIT_ASSERT_EQUAL(100, config.GetTestInt1());
        UNIT_ASSERT_EQUAL(0, config.GetTestInt2());
    }
    Y_UNIT_TEST(EnvAndFile) {
        SetEnv("TEST7_VAR", "100");

        TTempFile file("config.pb.txt");
        {
            TTestMessage conf;
            conf.SetTestString5("test_value_from_file");
            TString str;
            TFileOutput fout(file.Name());
            ::google::protobuf::TextFormat::PrintToString(conf, &str);
            fout << str;
        }

        const char* argv[] = {"./a.out", "--config", file.Name().c_str(), "--test-string6", "test_value_from_cmd"};
        auto config = NCrypta::ParsePbOptionsExtended<TTestMessage>(5, argv);

        UNIT_ASSERT_EQUAL(100, config.GetTestInt1());
        UNIT_ASSERT_EQUAL("test_value_from_file", config.GetTestString5());
        UNIT_ASSERT_EQUAL("test_value_from_cmd", config.GetTestString6());
    }

    void TestSelectFromEnvInt(const char* env, ui32 ref) {
        SetEnv("TEST9_VAR", env);
        auto config = NCrypta::ParsePbOptionsExtended<TTestMessage>(argc, argv);
        UNIT_ASSERT_EQUAL(ref, config.GetTestInt3());
    }

    Y_UNIT_TEST(SelectFromEnvInt) {
        TestSelectFromEnvInt("develop", 1);
        TestSelectFromEnvInt("testing", 2);
        TestSelectFromEnvInt("stable", 3);
    }

    Y_UNIT_TEST(FromEnvOneof) {
        SetEnv("TEST10_VAR", "TestString10Value");
        SetEnv("TEST13_VAR", "TestString13Value");

        auto config = NCrypta::ParsePbOptionsExtended<TOneofTestMessage>(argc, argv);
        UNIT_ASSERT_EQUAL("TestString10Value", config.GetA().GetField());
        UNIT_ASSERT_EQUAL("", config.GetB().GetField());
        UNIT_ASSERT_EQUAL("", config.GetC().GetField());
        UNIT_ASSERT_EQUAL("TestString13Value", config.GetD().GetField());
        UNIT_ASSERT_EQUAL("", config.GetE().GetField());
    }

    Y_UNIT_TEST(FromEnvOneofException) {
        SetEnv("TEST10_VAR", "TestString10Value");
        SetEnv("TEST11_VAR", "TestString11Value");

        UNIT_ASSERT_EXCEPTION_CONTAINS(NCrypta::ParsePbOptionsExtended<TOneofTestMessage>(argc, argv), yexception, "Two different fields in oneof are partially set by environment variables");
    }
}
