#include "synchronous_client.h"

#include <limits>

namespace NCrypta {
    const TSynchronousClient::TValueEncoder TSynchronousClient::IdentityEncoder = [](const TStringBuf value){return TString(value);};

    TSynchronousClient::TSynchronousClient(TValueEncoder valueEncoder, const TString &logName)
        : Database(nullptr)
        , ReplyHandler(nullptr)
        , Log(NLog::GetLog(logName))
        , ValueEncoder(valueEncoder) {
        ResetRequest();
    }

    TSynchronousClient::TSynchronousClient(IDatabasePtr database, IReplyHandlerPtr replyHandler, TValueEncoder valueEncoder, const TString &logName)
        : TSynchronousClient(valueEncoder, logName) {
        SetDatabase(database);
        SetReplyHandler(replyHandler);
    }

    void TSynchronousClient::SetDatabase(IDatabasePtr database) {
        Database = database;
    }

    void TSynchronousClient::SetReplyHandler(IReplyHandlerPtr replyHandler) {
        ReplyHandler = replyHandler;

        ReplyHandler->SetGetHandler([this](TDbRequestId id, TMaybe<NCrypta::TRecord> &&record) {
            if (id != RequestId) {
                LogUnexpectedRequestId(id);
            } else if (record) {
                Log->info("[{}] Get: {} : {} cas={}", id, record->Key, ValueEncoder(record->Value), record->Cas);
                Result = record;
            } else {
                Log->error("[{}] Not found", id);
            }

            ProcessRequest(id);
        });

        ReplyHandler->SetStoreHandler([this](TDbRequestId id, TString &&key) {
            if (id != RequestId) {
                LogUnexpectedRequestId(id);
            }
            Log->info("[{}] {} Store: OK", id, key);
            ProcessRequest(id);
        });

        ReplyHandler->SetRemoveHandler([this](TDbRequestId id, TString &&key) {
            if (id != RequestId) {
                LogUnexpectedRequestId(id);
            }
            Log->info("[{}] {} Remove: OK", id, key);
            ProcessRequest(id);
        });

        ReplyHandler->SetErrorHandler([this](TDbRequestId id, TString &&key, TString &&error) {
            if (id != RequestId) {
                LogUnexpectedRequestId(id);
            }
            Log->error("[{}] {} Error: {}", id, key, error);
            ProcessRequest(id);
        });

    }

    TMaybe<NCrypta::TRecord> TSynchronousClient::Get(const TString &key) {
        ResetRequest();
        RequestId = Database->Get(key, TDuration());
        Log->debug("[{}] Sending get: {}", RequestId, key);
        while (!IsRequestReplied) {
            ReplyHandler->TryProcessReply();
        }
        return Result;
    };

    void TSynchronousClient::Set(const TString &key, const TString &value, TDuration ttl, ui64 cas) {
        ResetRequest();
        RequestId = Database->Store({key, value, cas}, ttl);
        Log->debug("[{}] Sending store: {} : {} cas={}", RequestId, key, value, cas);
        while (!IsRequestReplied) {
            ReplyHandler->TryProcessReply();
        }
    }

    void TSynchronousClient::Remove(const TString &key, ui64 cas) {
        ResetRequest();
        RequestId = Database->Remove(key, cas);
        Log->debug("[{}] Sending remove: {} cas={}", RequestId, key, cas);
        while (!IsRequestReplied) {
            ReplyHandler->TryProcessReply();
        }
    }

    void TSynchronousClient::ResetRequest() {
        RequestId = std::numeric_limits<decltype(RequestId)>::max();
        Result.Clear();
        IsRequestReplied = false;
    }

    void TSynchronousClient::ProcessRequest(TDbRequestId id) {
        IsRequestReplied = true;
        Log->info("Processed request {}", id);
    };

    void TSynchronousClient::LogUnexpectedRequestId(TDbRequestId id) {
        Log->error("Unexpected request id {} instead of {}", id, RequestId);
    }
}
