#pragma once

#include <util/generic/hash.h>
#include <util/network/address.h>
#include <util/str_stl.h>

namespace NCrypta {
    /// Hash functor that takes an address and returns a hash value only for its
    /// IP address part, not taking port into account.
    struct TAddrOnlyHash {
        inline size_t operator()(const NAddr::TOpaqueAddr& s) {
            switch (s.Addr()->sa_family) {
                case AF_INET: {
                    auto in = reinterpret_cast<const sockaddr_in*>(s.Addr());
                    return THash<decltype(in->sin_addr.s_addr)>().operator()(in->sin_addr.s_addr);
                }
                case AF_INET6: {
                    auto in6 = reinterpret_cast<const sockaddr_in6*>(s.Addr());
                    return ComputeHash(TStringBuf{reinterpret_cast<const char *>(&in6->sin6_addr), sizeof(in6->sin6_addr)});
                }
                default:
                    return 0;
            }
        }
    };

    struct TAddrOnlyEqual {
        inline bool operator()(const NAddr::TOpaqueAddr& lhs, const NAddr::TOpaqueAddr& rhs) {
            if (lhs.Addr()->sa_family == rhs.Addr()->sa_family) {
                switch (lhs.Addr()->sa_family) {
                    case AF_INET: {
                        auto lhsIn = reinterpret_cast<const sockaddr_in*>(lhs.Addr());
                        auto rhsIn = reinterpret_cast<const sockaddr_in*>(rhs.Addr());
                        return lhsIn->sin_addr.s_addr == rhsIn->sin_addr.s_addr;
                    }
                    case AF_INET6: {
                        auto lhsIn6 = reinterpret_cast<const sockaddr_in6*>(lhs.Addr());
                        auto rhsIn6 = reinterpret_cast<const sockaddr_in6*>(rhs.Addr());
                        return !memcmp(&lhsIn6->sin6_addr, &rhsIn6->sin6_addr, sizeof(lhsIn6->sin6_addr));
                    }
                }
            }

            return false;
        }
    };
}
