#include "addr_hash.h"
#include "caching_reverse_dns_resolver.h"
#include "reverse_dns_resolver.h"
#include "types.h"

#include <crypta/lib/native/time/sleeper.h>

#include <util/system/rwlock.h>
#include <util/system/thread.h>
#include <util/thread/factory.h>

#include <atomic>

namespace NCrypta {
    TCachingReverseDnsResolver::TCachingReverseDnsResolver()
        : IsThreadRunning(true)
    {
        ResolverThread = SystemThreadFactory()->Run([this](){ ResolverRoutine(); });
    }

    TCachingReverseDnsResolver::~TCachingReverseDnsResolver() {
        IsThreadRunning.store(false);
        ResolverThread->Join();
    }

    TMaybe<TString> TCachingReverseDnsResolver::Resolve(const NDns::TAddr& addr) const {
        {
            TReadGuardBase<TLightRWLock> guard(CacheLock);
            if (auto it = Cache.find(addr); it != Cache.end()) {
                return it->second;
            }
        }

        RequestsQueue.Enqueue(addr);
        return TMaybe<TString>();
    }

    void TCachingReverseDnsResolver::ResolverRoutine() {
        TThread::SetCurrentThreadName("rdns-resolver");

        TSleeper sleeper(1, TDuration::MilliSeconds(100));

        TAddr addr;
        while (IsThreadRunning.load()) {
            if (RequestsQueue.Dequeue(addr)) {
                sleeper.Wake();

                {
                    // Consecutive dequeues can bring the same addresses in a row.
                    // Check if we have already resolved the address on one of previous iterations.
                    TReadGuardBase<TLightRWLock> guard(CacheLock);
                    if (Cache.contains(addr)) {
                        continue;
                    }
                }

                auto result = TReverseDnsResolver::Resolve(addr);
                if (result.Outcome == NDns::EReverseRequestOutcome::Ok) {
                    TWriteGuardBase<TLightRWLock> guard(CacheLock);
                    Cache[addr] = result.Name;
                }
            } else {
                sleeper.Sleep();
            }
        }
    }
}
