#include "worker.h"

#include <yt/yt/core/concurrency/delayed_executor.h>

#include "util/string/split.h"

using namespace NCrypta::NEventProcessing;

TWorker::TWorker(
    const IProcessorFactory& processorFactory,
    NPQ::TCookieQueue& cookiesToCommit,
    TStats& stats,
    TDuration retryTimeout
)
    : ProcessorFactory(processorFactory)
    , CookiesToCommit(cookiesToCommit)
    , Stats(stats)
    , RetryTimeout(retryTimeout)
    , Log(NLog::GetLog("worker"))
{}

void TWorker::Process(NPQ::TConsumer::TReadResult&& readResult) {
    for (const auto& lbMessage : readResult.Data) {
        for (const auto& event: StringSplitter(lbMessage).Split('\n').SkipEmpty()) {
            Stats.Count->Add("event.count.total");

            size_t retries = 0;
            while (!ProcessEvent(event)) {
                NYT::NConcurrency::TDelayedExecutor::WaitForDuration(RetryTimeout);
                ++retries;
            }

            Stats.Count->Add("event.count.processed");
            Stats.Percentile->Add("event.count.retries", retries);
        }
    }

    CookiesToCommit.Enqueue(std::move(readResult.EpochCookie));
}

bool TWorker::ProcessEvent(const TStringBuf& event) {
    IProcessorPtr processor = nullptr;

    try {
        processor = ProcessorFactory.Create(event);
        Stats.Count->Add("event.count.valid");
    } catch (const yexception& e) {
        Stats.Count->Add("event.count.invalid");
        Log->error("Invalid event '{}'. Exception: '{}'", event, e.what());
        return true;
    }

    if (processor == nullptr) {
        Stats.Count->Add("event.count.unsupported");
        Log->warn("Unsupported event '{}'", event);
        return true;
    }

    return processor->Process();
}
