#include "exp_system.h"

#include <ads/bsyeti/libs/experiments/ab_experiments_source.h>

using NBSYeti::TExperimentSystem;

namespace NCrypta::NExpSystem {
    TExperimentSystemManager::TExperimentSystemManager(const TExperimentSystemOptions& options)
        : Options(options)
        , FetcherPtr(MakeAtomicShared<TSandboxFetcher>(Options.GetUrl()))
        , ExperimentSystemPtr(FetchExperimentSystem())
    {
    }

    TExperimentSystemManager::TExperimentSystemManager(const TExperimentSystemOptions& options, TAtomicSharedPtr<IFetcher> fetcher)
        : Options(options)
        , FetcherPtr(fetcher)
        , ExperimentSystemPtr(FetchExperimentSystem())
    {
    }

    TExperimentSystemManager::TExpSystemPtr TExperimentSystemManager::GetExperimentSystem() const {
        TReadGuard readGuard(ExperimentSystemLock);
        return ExperimentSystemPtr;
    }

    bool TExperimentSystemManager::Update() {
        for (ui32 retries = 0;; ++retries) {
            try {
                auto newExperimentSystem = FetchExperimentSystem();
                {
                    TWriteGuard writeGuard(ExperimentSystemLock);
                    ExperimentSystemPtr = std::move(newExperimentSystem);
                }
                return true;
            } catch (const yexception& e) {
                if (retries >= Options.GetRetries()) {
                    throw;
                }
            }
        }
    }

    NBSYeti::NExperimentParameters::TBigbPlugin TExperimentSystemManager::GetBigbParameters(NExperiments::TUniqId id) const {
        return GetParameters<NBSYeti::NExperimentParameters::TBigbPlugin>(NExperiments::TUserIds{}.SetUniqId(id));
    }

    NBSYeti::NExperimentParameters::TEaglePlugin TExperimentSystemManager::GetEagleParameters(NExperiments::TUniqId id) const {
        return GetParameters<NBSYeti::NExperimentParameters::TEaglePlugin>(NExperiments::TUserIds{}.SetUniqId(id));
    }

    NBSYeti::NExperimentParameters::TBigbPlugin TExperimentSystemManager::GetBigbParameters(const NExperiments::TUserIds& id) const {
        return GetParameters<NBSYeti::NExperimentParameters::TBigbPlugin>(id);
    }

    NBSYeti::NExperimentParameters::TEaglePlugin TExperimentSystemManager::GetEagleParameters(const NExperiments::TUserIds& id) const {
        return GetParameters<NBSYeti::NExperimentParameters::TEaglePlugin>(id);
    }

    TExperimentSystemManager::TExpSystemPtr TExperimentSystemManager::FetchExperimentSystem() const {
        const auto& abResource = FetcherPtr->FetchAbSystem();

        TMemoryInput abResourceStream(abResource.data(), abResource.size());

        NExperiments::TParserConfig parserConfig = NExperiments::LEGACY_PARSING;
        // parserConfig.PluginsBlackList = {"TBsPlugin"}; // use parserConfig.PluginsWhiteList
        auto abExperimentsSource = NBSYeti::CreateAbExperimentsSource(abResourceStream,
                                                                      {
                                                                          .LongConfig = true,
                                                                          .ParserConfig = std::move(parserConfig),
                                                                          .RequireDefault = false,
                                                                      });

        TExperimentSystem expSystem;
        expSystem.AddExternalExperimentSource(abExperimentsSource);

        return MakeAtomicShared<NBSYeti::TExperimentSystem>(std::move(expSystem));
    }
}
