#include "stats_interceptor.h"

#include <crypta/lib/native/graphite/utils.h>
#include <crypta/lib/native/singleton/tagged_singleton.h>
#include <crypta/lib/native/time/utils.h>

#include <grpc++/server_context.h>
#include <contrib/libs/grpc/src/core/lib/channel/status_util.h>

#include <util/generic/strbuf.h>
#include <util/string/split.h>

using namespace NCrypta::NGrpc;

constexpr char METHOD_DELIMITER = '/';

namespace {
    TString GetHandle(const TStringBuf& method) {
        TStringBuf service;
        TString handle;
        StringSplitter(method).Split(METHOD_DELIMITER).SkipEmpty().Limit(2).TryCollectInto(&service, &handle);
        return handle;
    }
}

TStatsInterceptor::TStatsInterceptor(grpc::experimental::ServerRpcInfo* info, const TStats::TSettings& statsSettings)
    : Stats(TaggedSingleton<TStats, decltype(*this)>("grpc", statsSettings))
    , ArrivalTime(TInstant::Now())
    , Prefix(GetHandle(info->method()))
{
    Stats.Count->Add(MakeGraphiteMetric(Prefix, "request.total.received"));
}

void TStatsInterceptor::Intercept(grpc::experimental::InterceptorBatchMethods* methods) {
    if (methods->QueryInterceptionHookPoint(grpc::experimental::InterceptionHookPoints::POST_RECV_MESSAGE)) {
        StartProcessTime = TInstant::Now();
    }

    if (methods->QueryInterceptionHookPoint(grpc::experimental::InterceptionHookPoints::PRE_SEND_MESSAGE)) {
        FinishProcessing();
    }

    if (methods->QueryInterceptionHookPoint(grpc::experimental::InterceptionHookPoints::PRE_SEND_STATUS)) {
        Stats.Count->Add(MakeGraphiteMetric(
            MakeGraphiteMetric(Prefix, "request.status_code"),
            grpc_status_code_to_string(static_cast<grpc_status_code>(methods->GetSendStatus().error_code()))
        ));
    }

    if (methods->QueryInterceptionHookPoint(grpc::experimental::InterceptionHookPoints::POST_SEND_MESSAGE)) {
        if (methods->GetSendMessageStatus()) {
            Stats.Count->Add(MakeGraphiteMetric(Prefix, "request.total.replied"));
        }
    }

    if (methods->QueryInterceptionHookPoint(grpc::experimental::InterceptionHookPoints::POST_RECV_CLOSE)) {
        FinishProcessing();
        Stats.Percentile->Add(MakeGraphiteMetric(Prefix, "latency.wall_time"), MusecsTillNow(ArrivalTime));
    }

    methods->Proceed();
}

void TStatsInterceptor::FinishProcessing() {
    if (!ProcessingFinished) {
        ProcessingFinished = true;
        Stats.Percentile->Add(MakeGraphiteMetric(Prefix, "latency.process"), MusecsTillNow(StartProcessTime));
    }
}
