#include "log_entry_builder_factory.h"

#include <library/cpp/json/json_writer.h>
#include <library/cpp/string_utils/tskv_format/builder.h>

#include <util/generic/fwd.h>

using namespace NCrypta;

namespace {
    class TTskvLogEntryBuilder : public ILogEntryBuilder {
    public:
        TTskvLogEntryBuilder(const TString& logType)
            : Builder(logType) {
        }

        ILogEntryBuilder& Add(TStringBuf key, TStringBuf value) override {
            Builder.Add(key, value);
            return *this;
        }

        ILogEntryBuilder& Add(TStringBuf key, ui64 value) override {
            Builder.Add(key, ToString(value));
            return *this;
        }

        TString GetAndReset() override {
            TString result(std::move(Builder.Str()));
            Builder.Clear();
            return result;
        }

    private:
        NTskvFormat::TLogBuilder Builder;
    };

    class TTskvLogEntryBuilderFactory : public ILogEntryBuilderFactory {
    public:
        TTskvLogEntryBuilderFactory(const TString& logType)
            : LogType(logType) {
        }

        THolder<ILogEntryBuilder> Create() override {
            return MakeHolder<TTskvLogEntryBuilder>(LogType);
        }
    private:
        const TString LogType;
    };

    class TJsonLogEntryBuilder : public ILogEntryBuilder {
    public:
        TJsonLogEntryBuilder()
            : Writer(&Stream, false) {
            Writer.OpenMap();
        }

        ILogEntryBuilder& Add(TStringBuf key, TStringBuf value) override {
            Writer.Write(key, value);
            return *this;
        }

        ILogEntryBuilder& Add(TStringBuf key, ui64 value) override {
            Writer.Write(key, value);
            return *this;
        }

        TString GetAndReset() override {
            Writer.CloseMap();
            Writer.Flush();

            TString result(std::move(Stream.Str()));

            Stream.Clear();
            Writer.OpenMap();

            return result;
        }

    private:
        TStringStream Stream;
        NJson::TJsonWriter Writer;
    };

    class TJsonLogEntryBuilderFactory : public ILogEntryBuilderFactory {
    public:
        THolder<ILogEntryBuilder> Create() override {
            return MakeHolder<TJsonLogEntryBuilder>();
        }
    };
}

THolder<ILogEntryBuilderFactory> NCrypta::NewTskvLogEntryBuilderFactory(const TString& logType) {
    return MakeHolder<TTskvLogEntryBuilderFactory>(logType);
}

THolder<ILogEntryBuilderFactory> NCrypta::NewJsonLogEntryBuilderFactory() {
    return MakeHolder<TJsonLogEntryBuilderFactory>();
}
