#include "request_reply.h"

#include <crypta/lib/native/access_log/access_log_fields.h>
#include <crypta/lib/native/dns/fqdn_mask.h>
#include <crypta/lib/native/graphite/utils.h>
#include <crypta/lib/native/http/format.h>
#include <crypta/lib/native/log/log.h>
#include <crypta/lib/native/time/utils.h>
#include <crypta/lib/native/tvm/get_checked_service_ticket.h>
#include <crypta/lib/native/tvm/get_checked_user_ticket.h>

#include <library/cpp/string_utils/base64/base64.h>

#include <util/network/sock.h>
#include <util/string/builder.h>

using namespace NCrypta;
using namespace NCrypta::NHttp;

TRequestReply::TRequestReply(
        THolder<::NNeh::IHttpRequest> request,
        const NTvmAuth::TTvmClient& tvmClient,
        const TCachingReverseDnsResolver& reverseDnsResolver,
        THolder<ILogEntryBuilder>&& accessLogEntryBuilder,
        NPQ::TProducer& accessLogProducer,
        bool logBody,
        TStats& stats)
    : Request(std::move(request))
    , TvmClient(tvmClient)
    , ReverseDnsResolver(reverseDnsResolver)
    , AccessLogEntryBuilder(std::move(accessLogEntryBuilder))
    , AccessLogProducer(accessLogProducer)
    , LogBody(logBody)
    , Stats(stats)
{
}

TStringBuf TRequestReply::GetRequestService() {
    return Request->Service();
}

TStringBuf TRequestReply::GetRequestCgi() {
    return Request->Cgi();
}

TStringBuf TRequestReply::GetRequestBody() {
    return Request->Body();
}

TStringBuf TRequestReply::GetRequestMethod() {
    return Request->Method();
}

const THttpHeaders& TRequestReply::GetRequestHeaders() {
    return Request->Headers();
}

const TMaybe<NTvmAuth::TCheckedServiceTicket>& TRequestReply::GetClientServiceTicket() {
    if (!ServiceTicket.Defined()) {
        ParseTvmServiceTicket();
    }
    return *ServiceTicket;
}

const TMaybe<NTvmAuth::TCheckedUserTicket>& TRequestReply::GetUserTicket() {
    if (!UserTicket.Defined()) {
        ParseTvmUserTicket();
    }
    return *UserTicket;
}

NAddr::TOpaqueAddr TRequestReply::GetRemoteAddr() const {
    const auto& ip = Request->RemoteHost();
    if (ip.at(0) == '[') {
        TSockAddrInet6 sockAddr(ip.substr(1, ip.size()-2).c_str(), 0);
        return NAddr::TOpaqueAddr(sockAddr.SockAddr());
    } else {
        TSockAddrInet sockAddr(ip.c_str(), 0);
        return NAddr::TOpaqueAddr(sockAddr.SockAddr());
    }
}

void TRequestReply::WriteAccessLog(const HttpCodes& httpCode, const TString& body) {
    auto remoteAddr = GetRemoteAddr();
    auto resolvedAddr = ReverseDnsResolver.Resolve(remoteAddr);
    auto srcHost = resolvedAddr.Defined()
                   ? resolvedAddr.GetRef()
                   : (remoteAddr.Addr()->sa_family == AF_INET
                      ? "unresolved_ipv4"
                      : "unresolved_ipv6");

    TString clientTvmIdStr;
    const auto& serviceTicket = GetClientServiceTicket();
    if (serviceTicket.Defined()) {
        clientTvmIdStr = *serviceTicket ? ToString(serviceTicket->GetSrc()) : "invalid";
    }

    (*AccessLogEntryBuilder)
            .Add(NAccessLogFields::UNIXTIME, TInstant::Now().Seconds())
            .Add(NAccessLogFields::QUERY, TStringBuilder() << "/" << GetRequestService() << "?" << GetRequestCgi())
            .Add(NAccessLogFields::LATENCY, MusecsTillNow(Request->ArrivalTime()))
            .Add(NAccessLogFields::TS_MICRO, Request->ArrivalTime().MicroSeconds())
            .Add(NAccessLogFields::SRC_IP, NAddr::PrintHost(remoteAddr))
            .Add(NAccessLogFields::SRC_HOST, srcHost)
            .Add(NAccessLogFields::CLIENT_TVM_ID, clientTvmIdStr)
            .Add(NAccessLogFields::HTTP_CODE, httpCode)
            .Add(NAccessLogFields::REPLY_BODY, body);

    if (LogBody) {
        AccessLogEntryBuilder->Add(NAccessLogFields::BODY, Base64Encode(GetRequestBody()));
    }

    Stats.Count->Add(MakeGraphiteMetric("srcs", CanonizeGraphiteMetricNode(MaskHostNumbers(srcHost))));
    AccessLogProducer.TryEnqueue(AccessLogEntryBuilder->GetAndReset());
}

void TRequestReply::ParseTvmServiceTicket() {
    ServiceTicket = NCrypta::GetCheckedServiceTicket(TvmClient, GetRequestHeaders());
}

void TRequestReply::ParseTvmUserTicket() {
    UserTicket = NCrypta::GetCheckedUserTicket(TvmClient, GetRequestHeaders());
}

void TRequestReply::Reply(HttpCodes httpCode, const TString& response, TStats& stats, NLog::TLogPtr log) {
    if (Replied) {
        stats.Count->Add("errors.repeated_reply");
        return;
    }

    SendResponse(httpCode, response, log);

    stats.Count->Add("request.total.replied");
    stats.Count->Add("request.http_code." + ToString(static_cast<int>(httpCode)));
    stats.Percentile->Add("latency.wall_time", MusecsTillNow(Request->ArrivalTime()));
}

void TRequestReply::ReplyError(HttpCodes httpCode, const TString& errorMessage, TStats& stats, NLog::TLogPtr log) {
    log->error(errorMessage);
    Reply(httpCode, GetSimpleResponse(errorMessage), stats, log);
}

void TRequestReply::SendResponse(const HttpCodes& httpCode, const TString& body, NLog::TLogPtr& log) {
    try {
        ::NNeh::TDataSaver out;
        out << body;
        Request->SendReply(out, "", httpCode);
        Replied = true;
        WriteAccessLog(httpCode, body);
    } catch (const yexception& e) {
        log->error("Failed to send reply\ncode: {}\nbody:\n{}\nbecause of error: {}", httpCode, body, e.what());
    } catch (...) {
        log->error("Unknown exception");
    }
}

bool TRequestReply::IsReplied() {
    return Replied;
}

TInstant TRequestReply::GetStartTime() {
    return Request->ArrivalTime();
}
