#include "server.h"

#include "request_reply.h"

#include <crypta/lib/native/singleton/tagged_singleton.h>

#include <library/cpp/neh/http_common.h>
#include <library/cpp/neh/http2.h>
#include <library/cpp/neh/neh.h>
#include <library/cpp/neh/multi.h>

#include <util/string/builder.h>

using namespace NCrypta::NHttp;

void TServer::OnRequest(::NNeh::IRequestRef req) {
    ProcessThreadPool->GetInvoker()->Invoke(BIND([this, reqHolder = THolder<::NNeh::IHttpRequest>(dynamic_cast<::NNeh::IHttpRequest*>(req.Release()))]() mutable {
        TRequestReply requestReply(
            std::move(reqHolder),
            TvmClient,
            ReverseDnsResolver,
            AccessLogEntryBuilderFactory->Create(),
            AccessLogProducer,
            LogBody,
            Stats
        );

        TTryGuard<TFastSemaphore> guard(InFlightSemaphore);
        if (!guard.WasAcquired()) {
            requestReply.ReplyError(HttpCodes::HTTP_TOO_MANY_REQUESTS, "Too many requests", Stats, Log);
            return;
        }

        Func(requestReply);
    }));
}

TServer::TServer(
    const TFunc& func,
    const THttpConfig& config,
    const NTvmAuth::TTvmClient& tvmClient,
    const TCachingReverseDnsResolver& reverseDnsResolver,
    THolder<ILogEntryBuilderFactory> accessLogEntryBuilderFactory,
    NPQ::TProducer& accessLogProducer,
    const TStats::TSettings& statsSettings
)
    : ProcessThreadPool(NYT::New<NYT::NConcurrency::TThreadPool>(config.GetWorkerThreads(), "WorkerInvkr"))
    , Func(std::move(func))
    , InFlightSemaphore(config.GetMaxInFlightRequests())
    , LogBody(config.GetLogBody())
    , TvmClient(tvmClient)
    , ReverseDnsResolver(reverseDnsResolver)
    , AccessLogEntryBuilderFactory(std::move(accessLogEntryBuilderFactory))
    , AccessLogProducer(accessLogProducer)
    , Stats(TaggedSingleton<TStats, decltype(*this)>("server", statsSettings))
    , Log(NLog::GetLog("server"))
{
    Y_ENSURE(config.GetMaxInFlightRequests() > 0, "Max in flight requests must be > 0");
    Stats.Count->Add("start");

    ::NNeh::THttp2Options::LimitRequestsPerConnection = config.GetMaxRequestsPerConnection();
    ::NNeh::THttp2Options::AsioServerThreads = config.GetAsioServerThreads();
    Requester = ::NNeh::MultiRequester({TStringBuilder() << "http://localhost:" << config.GetPort()}, this);
}
