#include <cstddef>
#include <cstdint>

#include <util/random/random.h>
#include <util/stream/output.h>

#include <crypta/lib/native/identifiers/lib/generic.h>
#include <crypta/lib/native/identifiers/lib/id_types/all.h>

static const TString kDELIM = "$";
static const TVector<TString> kIDENTIFIER_KEYS = []() {
    TVector<TString> buff;
    buff.reserve(NIdentifiers::IDENTIFIERS_NAME_TO_TYPE.size());
    for (const auto& [key, value] : NIdentifiers::IDENTIFIERS_NAME_TO_TYPE) {
        buff.push_back(key);
    }
    return buff;
}();

extern "C" {
    size_t LLVMFuzzerCustomMutator(uint8_t* data, const size_t size, const size_t maxSize,
                                   const unsigned int seed) {
        if (maxSize < 32) {
            // protect from small input
            return 0;
        }

        SetRandomSeed(seed);
        Y_UNUSED(size);

        const auto idType = kIDENTIFIER_KEYS[RandomNumber<uint32_t>(kIDENTIFIER_KEYS.size())];
        const auto idProtoType = NIdentifiers::IDENTIFIERS_NAME_TO_TYPE.at(idType);
        const auto idValue = NIdentifiers::TGenericID::Next(idProtoType);

        const TString buffer = idType + kDELIM + idValue;

        if (buffer.size() > maxSize) {
            return 0;
        }

        memcpy(data, buffer.data(), buffer.size());
        return buffer.size();
    }

    int LLVMFuzzerTestOneInput(const uint8_t* data, const size_t size) {
        if (size <= 1) {
            // protect from uninitialized input
            return 0;
        }

        const TString idPair{reinterpret_cast<const char*>(data), size};

        const size_t delimIndex = idPair.find(kDELIM, 0);
        const auto idType = idPair.substr(0, delimIndex);
        const auto idValue = idPair.substr(delimIndex + 1, idPair.size());
        const auto idProtoType = NIdentifiers::IDENTIFIERS_NAME_TO_TYPE.at(idType);

        const NIdentifiers::TGenericID identifier{idType, idValue};
        const NIdentifiers::TGenericID fromProto{identifier.ToProto()};

        const bool isCorrect = identifier.IsValid() && fromProto.IsValid() &&
            (identifier.GetType() == idProtoType) && (identifier.GetValue() == idValue) &&
            (identifier.GetType() == fromProto.GetType()) && (
                // mm device id may be short (24) and long (32) symbols,
                // with dashes (-), UPPER or LOWER. So skip for MM_DEVICE_ID
                (idProtoType == NCrypta::NIdentifiersProto::NIdType::MM_DEVICE_ID) ||
                ((identifier.GetValue() == fromProto.GetValue()) &&
                 (identifier.Normalize() == fromProto.Normalize())));

        if (!isCorrect) {
            Cout << idType << " " << idValue << " ";
            if (!identifier.IsValid()) {
                Cout << "Identifier invalid";
            } else if (!fromProto.IsValid()) {
                Cout << "From proto invalid";
            } else if (identifier.GetType() != idProtoType) {
                Cout << "Incorrect proto Type";
            } else if (identifier.GetValue() != idValue) {
                Cout << "Incorrect Value " << identifier.GetValue();
            } else if (identifier.GetType() != fromProto.GetType()) {
                Cout << "Incorrect proto Type (from proto)";
            } else if (identifier.GetValue() != fromProto.GetValue()) {
                Cout << "Incorrect Value (from proto) " << fromProto.GetValue();
            } else if (identifier.Normalize() != fromProto.Normalize()) {
                Cout << "Incorrect Normalize (from proto) " << fromProto.Normalize();
            }
            Cout << Endl;
            __builtin_trap();
        }

        return 0;
    }
}
