#include "generic.h"

#include <crypta/lib/native/identifiers/lib/id_types/all.h>
#include <util/generic/yexception.h>

namespace NIdentifiers {

    TGenericID::TGenericID(const NCrypta::NIdentifiersProto::TGenericID& proto)
            : Type(proto.GetType()) {
        SetValue(proto);
    }

    TGenericID::TGenericID(const EIdType& type,
                           const TString& identifierValue)
            : Type(type) {
        SetValue(identifierValue);
    }

    TGenericID::TGenericID(const TString& identifierType, const TString& identifierValue)
        : TGenericID(GetIdentifierTypeByName(to_lower(identifierType)), identifierValue) {
    }

    TGenericID::TGenericID(const TGenericID& other)
        : TGenericID(other.GetType(), other.GetValue()) {
    }

    TGenericID::TGenericID(TGenericID&& other)
        : Type(other.Type)
        , Identifier(std::move(other.Identifier)) {
    }

    TString TGenericID::Next(const EIdType& identifierType) {
        if (const auto& iterator{NIdentifiers::IDENTIFIERS_NEXT_MAP.find(identifierType)};
            iterator == NIdentifiers::IDENTIFIERS_NEXT_MAP.end()) {
            ythrow yexception() << "identifier not found";
        } else {
            return iterator->second();
        }
    }

    template <class ValueType>
    void TGenericID::SetValue(const ValueType& value) {
        if (const auto& iterator{NIdentifiers::IDENTIFIERS_MAP<ValueType>.find(GetType())};
            iterator == NIdentifiers::IDENTIFIERS_MAP<ValueType>.end()) {
            Identifier = nullptr;
            ythrow yexception() << "identifier not found";
        } else {
            Identifier = iterator->second(value);
        }
    }

    void TGenericID::SetValue(const char* value) {
        // template deduced const char* from string literals but we need TString
        // so static cast to template
        SetValue(static_cast<TString>(value));
    }

    EIdType TGenericID::GetType() const {
        return Type;
    }

    TString TGenericID::GetTypeString() const {
        return NCrypta::NIdentifiersProto::NIdType::EIdType_descriptor()
            ->FindValueByNumber(static_cast<i32>(Type))
            ->options()
            .GetExtension(NCrypta::NIdentifiersProto::Name);
    }

    bool TGenericID::IsValid() const {
        if (Identifier == nullptr) {
            return false;
        }
        return Identifier->IsValid();
    }

    bool TGenericID::IsSignificant() const {
        if (Identifier == nullptr) {
            return false;
        }
        return Identifier->IsSignificant();
    }

    size_t TGenericID::Hash() const {
        if (Identifier == nullptr) {
            return 0;
        }
        return Identifier->Hash();
    }


    TString TGenericID::GetValue() const {
        if (Identifier == nullptr) {
            return "";
        }
        return Identifier->Value();
    }

    TString TGenericID::GetNext() const {
        if (Identifier == nullptr) {
            return "";
        }
        return Identifier->GetNext();
    }

    TString TGenericID::Normalize() const {
        if (Identifier == nullptr) {
            return "";
        }
        return Identifier->Normalize();
    }

    TString TGenericID::StrictNormalize() const {
        if (Identifier == nullptr) {
            return "";
        }
        return Identifier->StrictNormalize();
    }

    TString TGenericID::GetMd5() const {
        if (Identifier == nullptr) {
            ythrow yexception() << "identifier not found";
        }
        return Identifier->GetMd5();
    }

    TString TGenericID::GetSha256() const {
        if (Identifier == nullptr) {
            ythrow yexception() << "identifier not found";
        }
        return Identifier->GetSha256();
    }

    ui64 TGenericID::GetHalf() const {
        if (Identifier == nullptr) {
            ythrow yexception() << "identifier not found";
        }
        return Identifier->GetHalf();
    }

    bool TGenericID::HasLikelihood() const {
        if (Identifier == nullptr) {
            ythrow yexception() << "identifier not found";
        }
        return Identifier->HasLikelihood();
    };

    double TGenericID::GetFrequencySpread() {
        if (Identifier == nullptr) {
            ythrow yexception() << "identifier not found";
        }
        return Identifier->GetFrequencySpread();
    }

    double TGenericID::GetAlternating() {
        if (Identifier == nullptr) {
            ythrow yexception() << "identifier not found";
        }
        return Identifier->GetAlternating();
    }

    double TGenericID::GetPeriodicity() {
        if (Identifier == nullptr) {
            ythrow yexception() << "identifier not found";
        }
        return Identifier->GetPeriodicity();
    }

    double TGenericID::GetRepetitions() {
        if (Identifier == nullptr) {
            ythrow yexception() << "identifier not found";
        }
        return Identifier->GetRepetitions();
    }

    bool TGenericID::operator==(const TGenericID& other) const {
        return ((Identifier == nullptr) && (other.Identifier == nullptr)) ||
               ((Identifier != nullptr) && (other.Identifier != nullptr) &&
                    ((*Identifier) == (*other.Identifier)));
    }

    bool TGenericID::operator==(const TIdentifier& other) const {
        return (Identifier != nullptr) && ((*Identifier) == other);
    }

    NCrypta::NIdentifiersProto::TGenericID TGenericID::ToProto() const {
        if (Identifier == nullptr) {
            ythrow yexception() << "identifier not found";
        }
        return Identifier->ToProto();
    }

    TString TGenericID::Serialize() const {
        TString serialized;
        Y_PROTOBUF_SUPPRESS_NODISCARD ToProto().SerializeToString(&serialized);
        return serialized;
    }
}

bool operator==(const NIdentifiers::TIdentifier& other,
                const NIdentifiers::TGenericID& self) {
    return (self == other);
}
