#include "duid.h"

#include <util/random/random.h>
#include <util/string/cast.h>

namespace NIdentifiers {

    ui64 TDuid::GetTimestamp(ui64 value) {
        while (value >= 10'000'000'000ull) {
            // get top 10 digits
            value /= 10u;
        }
        return value;
    }

    ui64 TDuid::GetTimestamp(const TString& value) {
        return GetTimestamp(FromString<ui64>(value));
    }

    bool TDuid::Validate(const TString& value) {
        return IsUint64(value) &&
               IsValidTimestamp(GetTimestamp(value)) &&
               IsValidRegexp(value);
    }

    bool TDuid::Significate(const TString& value) {
        return Validate(value)
               && (CountSignificant(value) > 0)
               && value.size() >= MinimalSigninficantSize;
    }

    bool TDuid::Significate(const ui64 value) {
        return Validate(value)
               && (value != 0)
               && ToString(value).size() >= MinimalSigninficantSize;
    }

    bool TDuid::Validate(const ui64 value) {
        return IsValidTimestamp(GetTimestamp(value)) &&
           (value >= (10'000'000'000'000'000ull));  // l17 with ts
            // (value <= 9'999'999'999'999'999'999ull);             // l20
    }

    TString TDuid::Next() {
        auto random_ts = GenerateRandomTimestamp();
        TString random_id;

        do {
            auto random_number = RandomNumber<ui32>(4294967295);
            random_id = ToString(random_number);
        } while(random_id.size() < MinimalSigninficantSize - 10);

        if (random_ts[0] == '0') {
            random_ts[0] = '1';
            random_ts[1] = '5';
        }

        return random_ts + random_id;
    }

    NCrypta::NIdentifiersProto::TGenericID TDuid::ToProto() const {
        auto proto = TIdentifier::ToProto();
        if (!proto.HasRawValue()) {
            proto.MutableDuid()->SetValue(FromString<ui64>(Normalize()));
        }
        return proto;
    }

    TString TDuid::FromProto(const NCrypta::NIdentifiersProto::TGenericID& proto) {
        return proto.HasRawValue() ? proto.GetRawValue() : ToString(proto.GetDuid().GetValue());
    }
}
