#include "email.h"

#include <util/generic/string.h>
#include <util/generic/vector.h>
#include <util/random/random.h>
#include <util/string/split.h>

#include <crypta/lib/native/identifiers/lib/utils.h>

namespace NIdentifiers {

    static const TVector<TString> DOMAINS{"yandex.ru", "gmail.com", "mail.ru", "rambler.ru"};

    void ModifyEmail(TString& email) {
        auto [local, domain] = SplitEmail(email);
        re2::RE2::GlobalReplace(&local, "(\\.)+", ".");

        if (const size_t indexAt{local.find("+", 0)}; indexAt != std::string::npos) {
            local = local.substr(0, indexAt);
        }

        if (domain == "yandex.ru") {
            SubstGlobal(local, ".", "-", 0);
        } else if (domain == "gmail.com") {
            SubstGlobal(local, ".", "", 0);
        }

        email = local + "@" + domain;
    }

    TEmail::TEmail(const TString& value)
            : TTypedIdentifier(value) {
        Original = StripStringLeft(StripString(to_lower(value)), TEqualsStripAdapter('+'));
    }

    TEmail::TEmail(const NCrypta::NIdentifiersProto::TGenericID& proto)
        : TEmail(FromProto(proto)) {
    }

    bool TEmail::Validate(const TString& value) {
        auto [local, domain] = SplitEmail(value);
        return RE2::FullMatch(value, COMMON_EMAIL_REGEXP) &&
               (!IsUnnormalEmail(value) || IsValidHeuristic(value)) &&
               (local.length() <= MAX_LOCAL_LENGTH) && (domain.length() <= MAX_DOMAIN_LENGTH);
    }

    TString TEmail::DoNormalize() const {
        TString normalized{Original};
        // googlemail.com to gmail.com
        re2::RE2::Replace(&normalized, GOOGLE_EMAIL_REGEXP, "@gmail.com");
        // all yandex mail's alias to yandex.ru
        if (IsValidHeuristic(Original)) {
            re2::RE2::Replace(&normalized, YANDEX_TLD_REGEXP, "@yandex.ru");
        }

        ModifyEmail(normalized);
        return normalized;
    }

    bool TEmail::IsUnnormalEmail(const TString& value) {
        return RE2::FullMatch(value, UNNORMAL_EMAIL_REGEXP) && !(
            RE2::FullMatch(value, YANDEX_DEFAULT_EMAIL_REGEXP)
            || RE2::FullMatch(value, YANDEX_TEAM_REGEXP));
    }

    bool TEmail::IsValidHeuristic(const TString& value) {
        return RE2::FullMatch(value, YANDEX_EMAIL_REGEXP);
    }

    TString TEmail::Next() {
        size_t email_len = 5u + RandomNumber<size_t>(5);
        TString email;
        while(email_len--) {
            email.append(NextAlphaNumeric());
        }
        return email + "@" + DOMAINS.at(RandomNumber<size_t>(DOMAINS.size()));
    }

    NCrypta::NIdentifiersProto::TGenericID TEmail::ToProto() const {
        auto proto = TIdentifier::ToProto();
        if (!proto.HasRawValue()) {
            const auto& [local, domain] = SplitEmail(Normalize());
            proto.MutableEmail()->SetLogin(local);
            proto.MutableEmail()->SetDomain(domain);
        }
        return proto;
    }

    TString TEmail::FromProto(const NCrypta::NIdentifiersProto::TGenericID& proto) {
        return proto.HasRawValue() ? proto.GetRawValue() : proto.GetEmail().GetLogin() + "@" + proto.GetEmail().GetDomain();
    }

    bool TEmail::HasLikelihood() const {
        return true;
    }

    TString TEmail::MakeClearRepresentation() const {
        auto [Login, Domain] = SplitEmail(Normalize());
        SubstGlobal(Login, "-", "", 0);
        return Login;
    }
}
