#include "hostname.h"

#include <util/generic/hash_set.h>
#include <util/generic/vector.h>
#include <util/random/random.h>
#include <util/string/ascii.h>

namespace {
    const static TString::value_type DELIM_CHAR{'.'};
    const static TStringBuf DELIM{&DELIM_CHAR, 1};

    bool verifyLabel(TStringBuf label) {
        const static TString::value_type HYP{'-'};
        const static size_t MAX_LABEL_LENGTH{63};
        const static size_t MIN_LABEL_LENGTH{1};

        if (label.size() > MAX_LABEL_LENGTH || label.size() < MIN_LABEL_LENGTH) {
            return false;
        }

        if (!IsAsciiAlpha(label.front()) || !IsAsciiAlnum(label.back())) {
            return false;
        }

        // TODO: add international domain validation
        for (size_t i = 1; i + 1 < label.size(); ++i) {
            if (!IsAsciiAlnum(label[i]) && label[i] != HYP) {
                return false;
            }
        }

        return true;
    }

    bool verifyTld(TStringBuf tld) {
        // TODO: should check tld over the list of known tlds?
        const static size_t MIN_TLD_LABEL_LENGTH{2};

        return tld.size() >= MIN_TLD_LABEL_LENGTH && verifyLabel(tld);
    }

    bool verifyDomain(TStringBuf domain) {
        const static size_t MAX_DOMAIN_LENGTH{255}; // https://datatracker.ietf.org/doc/html/rfc1035

        if (domain.size() > MAX_DOMAIN_LENGTH) {
            return false;
        }

        domain.ChopSuffix(DELIM); // allow 'example.com.'

        int domainLevel{0};
        for (TStringBuf label; domain.TrySplit(DELIM, label, domain);) {
            if (!verifyLabel(label)) {
                return false;
            }
            ++domainLevel;
        }

        if (domainLevel < 1) {
            return false;
        }

        return verifyTld(domain);
    }
}

namespace NIdentifiers {
    bool THostname::IsYandex(const TString& value) {
        return Validate(value) && RE2::FullMatch(value, YANDEX_DOMAIN_RE);
    }

    bool THostname::IsYastatic(const TString& value) {
        return YASTATIC == value;
    }

    bool THostname::Validate(const TString& value) {
        return verifyDomain(value);
    }

    // YanDex.ru. -> yandex.ru
    TString THostname::DoNormalize() const {
        TString normalized = Original;

        normalized.to_lower();

        if (!normalized.empty() && normalized.back() == DELIM_CHAR) {
            normalized.resize(normalized.size() - 1);
        }

        return normalized;
    }

    bool THostname::Significate(const TString& value) {
        if (!Validate(value)) {
            return false;
        };

        const static THashSet<TString> reservedTlds{
            "test",
            "example",
            "invalid",
            "localhost",
            "localdomain",
        };

        if (TStringBuf tld, tmp{value}; tmp.TryRSplit(DELIM, tmp, tld) && reservedTlds.contains(tld)) {
            return false;
        }

        const static re2::RE2 isReservedDomainRe(R"(^([0-9A-Za-z\.\-]+\.)?example\.(com|net|org)$)");

        return !RE2::FullMatch(value, isReservedDomainRe);
    }

    TString THostname::Next() {
        const static TVector<TString> tlds{"ru", "com.tr", "de", "yandex", "net"};
        const static TVector<TString> middle{"yandex", "gmail", "news", "avito"};
        const static TVector<TString> front{"www", "smtp", "disk", ""};

        auto choice = [](const TVector<TString>& v) {
            return std::cref(v[RandomNumber(v.size())]);
        };

        TString result;

        result.append(choice(front));
        result.append(result.size() ? DELIM : "");
        result.append(choice(middle));
        result.append(DELIM);
        result.append(choice(tlds));

        return result;
    }

    NCrypta::NIdentifiersProto::TGenericID THostname::ToProto() const {
        auto proto = TIdentifier::ToProto();
        if (!proto.HasRawValue()) {
            proto.MutableHostname()->SetValue(Normalize());
        }
        return proto;
    }

    TString THostname::FromProto(const NCrypta::NIdentifiersProto::TGenericID& proto) {
        return proto.HasRawValue() ? proto.GetRawValue() : proto.GetHostname().GetValue();
    }
}
