#include "icookie.h"

#include <util/datetime/base.h>
#include <util/random/random.h>
#include <util/stream/str.h>
#include <util/string/cast.h>

namespace NIdentifiers {

    bool TIcookie::IsValidRegexp(const TString& value) {
        return RE2::FullMatch(value, ICOOKIE_REGEXP);
    }

    bool TIcookie::IsValidTimestamp(const ui64 timestamp) {
        return TCookieMixin::IsValidTimestamp(timestamp)
                || (timestamp <= UUID_ICOOKIE_MAX_TIMESTAMP);
    }

    bool TIcookie::Validate(const TString& value) {
        return IsUint64(value) &&
               IsValidTimestamp(GetTimestamp(value)) &&
               IsValidRegexp(value);
    }

    bool TIcookie::Validate(const ui64 value) {
        return IsValidTimestamp(GetTimestamp(value)) &&
               ((value >= 10'000'000'000'000'000ull) &&
               (value <= 9'999'999'999'999'999'999ull));
    }

    bool TIcookie::Significate(const ui64 value) {
        return Validate(value) && (value != 0);
    }

    ui64 TIcookie::GetTimestamp() const {
        return GetTimestamp(Original);
    }

    NCrypta::NIdentifiersProto::TGenericID TIcookie::ToProto() const {
        auto proto = TIdentifier::ToProto();
        if (!proto.HasRawValue()) {
            proto.MutableIcookie()->SetValue(FromString<ui64>(Normalize()));
        }
        return proto;
    }

    TString TIcookie::FromProto(const NCrypta::NIdentifiersProto::TGenericID& proto) {
        return proto.HasRawValue() ? proto.GetRawValue() : ToString(proto.GetIcookie().GetValue());
    }
}
