#include "all.h"
#include "idfa_gaid.h"
#include "gaid.h"
#include "idfa.h"

#include <util/generic/yexception.h>
#include <util/random/random.h>

namespace NIdentifiers {
    TIdfaGaid::TIdfaGaid(const TString& value)
            : TIdentifier(value, NCrypta::NIdentifiersProto::NIdType::IDFA_GAID) {
        if (IsLikeGaid(value)) {
            Identifier = IDENTIFIERS_MAP<TString>.at(NCrypta::NIdentifiersProto::NIdType::GAID)(value);
        } else if (IsLikeIdfa(value)) {
            Identifier = IDENTIFIERS_MAP<TString>.at(NCrypta::NIdentifiersProto::NIdType::IDFA)(value);
        } else {
            Identifier = nullptr;
        }
    }

    TIdfaGaid::TIdfaGaid(const NCrypta::NIdentifiersProto::TGenericID& proto)
            : TIdentifier(FromProto(proto), NCrypta::NIdentifiersProto::NIdType::IDFA_GAID) {
        if (proto.HasGaid()) {
            Identifier = IDENTIFIERS_MAP<NCrypta::NIdentifiersProto::TGenericID>.at(
                                           NCrypta::NIdentifiersProto::NIdType::GAID)(proto);
        } else if (proto.HasIdfa()) {
            Identifier = IDENTIFIERS_MAP<NCrypta::NIdentifiersProto::TGenericID>.at(
                                           NCrypta::NIdentifiersProto::NIdType::IDFA)(proto);
        } else {
            ythrow yexception() << "Incorrect proto";
        }
    }

    TIdfaGaid::TIdfaGaid(const TIdfaGaid& other)
        : TIdfaGaid(other.Identifier != nullptr ? other.Identifier->Value() : "") {
    } 

    bool TIdfaGaid::Validate() const {
        return Validate(Value());
    }

    bool TIdfaGaid::Validate(const TString& value) {
        return IsLikeGaid(value) || IsLikeIdfa(value);
    }

    bool TIdfaGaid::Significate(const TString& value) {
        if (IsLikeGaid(value)) {
            return TGaid::Significate(value);
        } else if (IsLikeIdfa(value)) {
            return TIdfa::Significate(value);
        } else {
            return false;
        }
    }

    TString TIdfaGaid::DoNormalize() const {
        if (Identifier != nullptr) {
            return Identifier->Normalize();
        }
        return "";
    }

    TString TIdfaGaid::Next() {
        return RandomNumber<size_t>(2) ? TGaid::Next() : TIdfa::Next();
    }

    NCrypta::NIdentifiersProto::TGenericID TIdfaGaid::ToProto() const {
        if (Identifier == nullptr) {
            ythrow yexception() << "Incorrect identifier";
        }
        auto proto = Identifier->ToProto();
        proto.SetType(GetType());  // keep IDFA_GAID as type of proto
        return proto;
    }

    bool TIdfaGaid::IsLikeGaid(const TString& value) {
        return RE2::FullMatch(value, GAID_REGEXP) && (value == to_lower(value));
    }

    bool TIdfaGaid::IsLikeIdfa(const TString& value) {
        return RE2::FullMatch(value, IDFA_REGEXP) && (value == to_upper(value));
    }

    bool TIdfaGaid::HasLikelihood() const {
        return true;
    }

    TString TIdfaGaid::FromProto(const NCrypta::NIdentifiersProto::TGenericID& proto) {
        THolder<TIdentifier> Identifier;
        if (proto.HasGaid()) {
            Identifier = IDENTIFIERS_MAP<NCrypta::NIdentifiersProto::TGenericID>.at(
                    NCrypta::NIdentifiersProto::NIdType::GAID)(proto);
        } else if (proto.HasIdfa()) {
            Identifier = IDENTIFIERS_MAP<NCrypta::NIdentifiersProto::TGenericID>.at(
                    NCrypta::NIdentifiersProto::NIdType::IDFA)(proto);
        } else {
            ythrow yexception() << "Incorrect proto";
        }
        return Identifier->Value();
    }
}
