#include "email.h"
#include "gaid.h"
#include "login.h"

#include <util/random/random.h>
#include <util/string/strip.h>
#include <util/string/split.h>

#include <crypta/lib/native/identifiers/lib/utils.h>

namespace NIdentifiers {
    bool TLogin::Validate(const TString& value) {
        if (RE2::FullMatch(value, TWO_OR_MORE_AT_REGEXP)) {
            return false;
        } else if (TEmail(value).IsValid()) {
            return true;
        } else if (value.length() > MAX_LOGIN_LENGTH ||
                value.length() == 0) {
            return false;
        } else if (value[0] >= '0' && value[0] <= '9') {
            return false;
        } else if (value[0] == '-' || value[0] == '.') {
            return false;
        } else if (value[value.length() - 1] == '-' ||
                   value[value.length() - 1] == '.') {
            return false;
        } else if (RE2::FullMatch(value, TWO_POINT_OR_HYPHEN_REGEXP)) {
            return false;
        } else if (TGaid(value).IsValid()) {
            return false;
        }
        return RE2::FullMatch(value, POSSIBLE_CHARS_REGEXP);
    }

    TString TLogin::DoNormalize() const {
        TString normalized = Original;
        if (normalized.find('@') == TString::npos) {
            SubstGlobal(normalized, ".", "-", 0);
        } else {
            normalized = NIdentifiers::TEmail(normalized).Normalize();
            if (RE2::FullMatch(normalized, YANDEX_EMAIL_REGEXP) ||
                RE2::FullMatch(normalized, YANDEX_DEFAULT_EMAIL_REGEXP)) {
                normalized = normalized.substr(0, normalized.find('@'));
            }
        }
        return normalized;
    }

    TString TLogin::MakeClearRepresentation() const {
        auto [Login, Domain] = SplitEmail(Original); // if not yandex
        SubstGlobal(Login, "-", "");
        return Login;
    }

    TString TLogin::Next() {
        size_t output_len = 5u + RandomNumber<size_t>(5);
        TString output(NextAlpha());
        while(output.size() < output_len) {
            output.append(NextAlphaNumeric());
        }
        return output;
    }

    NCrypta::NIdentifiersProto::TGenericID TLogin::ToProto() const {
        auto proto = TIdentifier::ToProto();
        if (!proto.HasRawValue()) {
            proto.MutableLogin()->SetValue(Normalize());
        }
        return proto;
    }

    bool TLogin::HasLikelihood() const {
        return true;
    }

    TString TLogin::FromProto(const NCrypta::NIdentifiersProto::TGenericID& proto) {
        return proto.HasRawValue() ? proto.GetRawValue() : proto.GetLogin().GetValue();
    }
}
