#pragma once

#include <crypta/lib/native/identifiers/lib/identifier.h>

#include <contrib/libs/re2/re2/re2.h>

static const re2::RE2 MD5_REGEXP("[0-9a-fA-F]{32}");

namespace NIdentifiers {

    template <class Identifier, EIdType idType>
    class TMd5Base : public TTypedIdentifier<Identifier, idType> {
    public:
        using TTypedIdentifier<Identifier, idType>::TTypedIdentifier;

        static TString Next() {
            return ::ComputeMd5Hash(NextNonZeroUInt());
        }

        static TString FromProto(const NCrypta::NIdentifiersProto::TGenericID& proto) {
            return proto.HasRawValue() ? proto.GetRawValue() : UIntToHex(proto.GetMd5().GetValue());
        }

        TString GetNext() const override {
            return Next();
        }

        NCrypta::NIdentifiersProto::TGenericID ToProto() const override {
            auto proto = TIdentifier::ToProto();
            if (!proto.HasRawValue()) {
                auto proto_big_int = HexToUInt128(Identifier::Normalize());
                proto.MutableMd5()->MutableValue()->SetLo(proto_big_int.GetLo());
                proto.MutableMd5()->MutableValue()->SetHi(proto_big_int.GetHi());
            }
            return proto;
        };

        static bool Validate(const TString& value) {
            return RE2::FullMatch(value, MD5_REGEXP);
        }

    protected:
        TString DoNormalize() const override {
            return to_lower(Identifier::Original);
        }
    };

    class TMd5 : public TMd5Base<TMd5, NCrypta::NIdentifiersProto::NIdType::MD5> {
    public:
        using TMd5Base::TMd5Base;
    };

    class TEmailMd5 : public TMd5Base<TEmailMd5, NCrypta::NIdentifiersProto::NIdType::EMAIL_MD5> {
    public:
        using TMd5Base::TMd5Base;
    };

    class TPhoneMd5 : public TMd5Base<TPhoneMd5, NCrypta::NIdentifiersProto::NIdType::PHONE_MD5> {
    public:
        using TMd5Base::TMd5Base;
    };

    class TMacExtMd5 : public TMd5Base<TMacExtMd5, NCrypta::NIdentifiersProto::NIdType::MAC_EXT_MD5> {
    // Special type for not canonical md5 hash from external users (https://st.yandex-team.ru/BIGB-477)
    public:
        using TMd5Base::TMd5Base;

        static TString Next();
        static TString FromProto(const NCrypta::NIdentifiersProto::TGenericID& proto);
    protected:
        TString DoNormalize() const override;
    };
}
