#include "gaid.h"
#include "idfa.h"
#include "mm_device_id.h"

#include <util/string/subst.h>
#include <util/generic/hash_set.h>
#include <util/random/random.h>

namespace NIdentifiers {
    bool TMmDeviceId::Validate(const TString& value) {
        return RE2::FullMatch(value, MM_DEVICE_ID_REGEXP) ||
               RE2::FullMatch(value, SHORT_MM_DEVICE_ID_REGEXP);
    }

    TString TMmDeviceId::DoNormalize() const {
        TString normalized = Original;
        if (RE2::FullMatch(Original, SHORT_MM_DEVICE_ID_REGEXP)) {
            normalized.to_lower();
            size_t leed_zeros = 32 - normalized.size();
            while (leed_zeros != 0) {
                normalized.prepend('0');
                --leed_zeros;
            }
        }
        // Carousel with lower/upper dashes
        else if (RE2::FullMatch(Original, LOWER_MM_REGEXP) || RE2::FullMatch(Original, ONLY_DIGITS_REGEXP)) {
            if (normalized.size() > 32) {
                SubstGlobal(normalized, "-", "", 0);
            }
        } else if (RE2::FullMatch(Original, UPPER_MM_REGEXP)) {
            if (normalized.size() == 32) {
                normalized = HexDigestDashes(normalized);
            }
        } else {
            SubstGlobal(normalized, "-", "", 0);
            normalized.to_lower();
        }
        return normalized;
    }

    TString TMmDeviceId::DoStrictNormalize() const {
        auto normalized{DoNormalize()};
        SubstGlobal(normalized, "-", "", 0);
        normalized.to_lower();
        return normalized;
    }

    TString TMmDeviceId::Next() {
        switch (size_t kind = RandomNumber<size_t>(5)) {
            case 0:
                // upper with dashes
                return TIdfa::Next();
            case 1:
                // lower with dashes
                return TGaid::Next();
            case 2: {
                // short mm_device_id
                TString output;
                while(output.size() < 24) {
                    output.append(NextHexDigest());
                }
                return output;
            }
            default: {
                // 32 hexdigest
                TString output;
                while (output.size() < 32) {
                    output.append(NextHexDigest());
                }
                if (kind & 1) {
                    // kind == 5; 32 hexdigest all upper
                    output.to_upper();
                } else {
                    // kind == 4; 32 hexdigest all lower
                    output.to_lower();
                }
                return output;
            }
        }
    }

    NCrypta::NIdentifiersProto::TGenericID TMmDeviceId::ToProto() const {
        auto proto = TIdentifier::ToProto();
        if (!proto.HasRawValue()) {
            auto proto_big_int = HexToUInt128(Normalize());
            proto.MutableMmDeviceId()->MutableValue()->SetLo(proto_big_int.GetLo());
            proto.MutableMmDeviceId()->MutableValue()->SetHi(proto_big_int.GetHi());
        }
        return proto;
    }

    bool TMmDeviceId::HasLikelihood() const {
        return true;
    }

    TString TMmDeviceId::FromProto(const NCrypta::NIdentifiersProto::TGenericID& proto) {
        return proto.HasRawValue() ? proto.GetRawValue() : UIntToHex(proto.GetMmDeviceId().GetValue());
    }
}
