#include "phone.h"

#include <util/random/random.h>
#include <util/string/cast.h>

using namespace i18n::phonenumbers;

namespace NIdentifiers {
    TPhone::TPhone(const TString& value)
        : TTypedIdentifier(value)
        , Phone(MakeHolder<PhoneNumber>()) {
    }

    TPhone::TPhone(const NCrypta::NIdentifiersProto::TGenericID& proto) : TPhone(FromProto(proto)) {
    }

    TPhone::TPhone(const TPhone& other)
        : TPhone(other.Original) {
    }

    bool TPhone::Validate(const TString& value) {
        if (!RE2::FullMatch(value, ALLOWED_CHARS_REGEXP)) {
            return false;
        }
        const auto phonePtr{MakeHolder<PhoneNumber>()};
        const bool isValid{
            (PhoneNumberUtil::NO_PARSING_ERROR == PhoneNumberUtil::GetInstance()->Parse(value, "RU", phonePtr.Get()))
            || (PhoneNumberUtil::NO_PARSING_ERROR == PhoneNumberUtil::GetInstance()->Parse("+" + value, "RU", phonePtr.Get()))
        };
        return isValid;
    }

    TString TPhone::DoNormalize() const {
        const auto consistNormalize{[&](const TString& value) {
                {
                    const bool isValid{
                        (PhoneNumberUtil::NO_PARSING_ERROR == PhoneNumberUtil::GetInstance()->Parse(value, "RU", Phone.Get()))
                        || (PhoneNumberUtil::NO_PARSING_ERROR == PhoneNumberUtil::GetInstance()->Parse("+" + value, "RU", Phone.Get()))
                    };
                    Y_UNUSED(isValid);
                }

                std::string result;
                PhoneNumberUtil::GetInstance()->Format(*Phone.Get(), PhoneNumberUtil::PhoneNumberFormat::E164, &result);
                return TString{result};
            }
        };

        TString prev{};
        TString next{consistNormalize(Original)};

        while (prev != next) {
            prev = next;
            next = consistNormalize(next);
        }
        return next;
    }

    TString TPhone::ComputeMd5Hash() const {
        return ::ComputeMd5Hash(Normalize().substr(1));
    }

    TString TPhone::ComputeSha256Hash() const {
        return ::ComputeSha256Hash(Normalize().substr(1));
    }

    TString TPhone::Next() {
        return "+" + ToString<ui64>(79'000'000'000ull + RandomNumber<ui64>(999'999'999ull));
    }

    NCrypta::NIdentifiersProto::TGenericID TPhone::ToProto() const {
        auto proto = TIdentifier::ToProto();
        if (!proto.HasRawValue()) {
            proto.MutablePhone()->SetValue(FromString<ui64>(Normalize().substr(1)));
        }
        return proto;
    }

    TString TPhone::FromProto(const NCrypta::NIdentifiersProto::TGenericID& proto) {
        return proto.HasRawValue() ? proto.GetRawValue() : "+" + ToString(proto.GetPhone().GetValue());
    }
}
