#pragma once

#include <crypta/lib/native/identifiers/lib/identifier.h>

#include <contrib/libs/re2/re2/re2.h>

static const re2::RE2 SHA256_REGEXP("[0-9a-fA-F]{64}");

namespace NIdentifiers {

    template <class Identifier, EIdType idType>
    class TSha256Base: public TTypedIdentifier<Identifier, idType> {
    public:
        using TTypedIdentifier<Identifier, idType>::TTypedIdentifier;

        static TString Next() {
            return ::ComputeSha256Hash(NextNonZeroUInt());
        }

        TString GetNext() const override {
            return Next();
        }

        NCrypta::NIdentifiersProto::TGenericID ToProto() const override {
            auto proto = TIdentifier::ToProto();
            if (!proto.HasRawValue()) {
                auto proto_big_int = HexToUInt256(Identifier::Normalize());
                proto.MutableSha256()->MutableValue()->SetV0(proto_big_int.GetV0());
                proto.MutableSha256()->MutableValue()->SetV1(proto_big_int.GetV1());
                proto.MutableSha256()->MutableValue()->SetV2(proto_big_int.GetV2());
                proto.MutableSha256()->MutableValue()->SetV3(proto_big_int.GetV3());
            }
            return proto;
        }

        static TString FromProto(const NCrypta::NIdentifiersProto::TGenericID& proto) {
            return proto.HasRawValue() ? proto.GetRawValue() : UIntToHex(proto.GetSha256().GetValue());
        }

        static bool Validate(const TString& value) {
           return RE2::FullMatch(value, SHA256_REGEXP);
        }
    protected:
        TString DoNormalize() const override {
            return to_lower(Identifier::Original);
        }
    };

    class TSha256 : public TSha256Base<TSha256, EIdType::SHA256> {
    public:
        using TSha256Base::TSha256Base;
    };

    class TEmailSha256 : public TSha256Base<TEmailSha256, EIdType::EMAIL_SHA256> {
    public:
        using TSha256Base::TSha256Base;
    };

    class TPhoneSha256 : public TSha256Base<TPhoneSha256, EIdType::PHONE_SHA256> {
    public:
        using TSha256Base::TSha256Base;
    };
}
