#include "ssp.h"

#include <util/string/cast.h>
#include <util/random/random.h>

namespace NIdentifiers {
    bool TSspUserId::Validate(const TString& value) {
        return RE2::FullMatch(value, SSP_USER_ID_REGEXP);
    }

    bool TSspUserId::Significate(const TString& value) {
        if (!Validate(value)) {
            return false;
        }

        ui64 sspid;
        TString ssp_user;
        RE2::FullMatch(value, SSP_USER_ID_REGEXP, &sspid, &ssp_user);
        return sspid > 10000 && CountSignificant(ssp_user) > 0;
    }

    TString TSspUserId::DoNormalize() const {
        TString normalized{Original};
        normalized.to_upper();
        return normalized;
    }

    TString TSspUserId::Next() {
        TString output;
        size_t id_len = 3 + RandomNumber<size_t>(5);
        while (output.size() < id_len) {
            output.append(NextAlpha());
        }
        return "10001@" + to_upper(output);
    }

    NCrypta::NIdentifiersProto::TGenericID TSspUserId::ToProto() const {
        auto proto = TIdentifier::ToProto();
        if (!proto.HasRawValue()) {
            const auto& [sspid_str, userid_str] = SplitEmail(Normalize());
            const ui64 sspid = FromString<ui64>(sspid_str);
            proto.MutableSspUserId()->SetSspid(sspid);
            proto.MutableSspUserId()->SetUserid(userid_str);
        }
        return proto;
    }

    TString TSspUserId::FromProto(const NCrypta::NIdentifiersProto::TGenericID& proto) {
        return proto.HasRawValue() ? proto.GetRawValue() : TStringBuilder() <<
            proto.GetSspUserId().GetSspid() << "@" << proto.GetSspUserId().GetUserid();
    }
}

