#include "yandexuid.h"

#include <util/datetime/base.h>
#include <util/random/random.h>
#include <util/stream/str.h>
#include <util/string/cast.h>

namespace NIdentifiers {
    ui64 TCookieMixin::GetMaxTimestamp() {
        return static_cast<ui64>(TInstant::Now().Seconds()) + YEAR_IN_SECONDS;
    }

    ui64 TCookieMixin::GetTimestamp(ui64 value) {
        return value % 10'000'000'000ull;
    }

    ui64 TCookieMixin::GetTimestamp(const TString& value) {
        return GetTimestamp(FromString<ui64>(value));
    }

    bool TCookieMixin::IsValidTimestamp(const ui64 timestamp) {
        return (timestamp >= MIN_TIMESTAMP) &&
               (timestamp <= GetMaxTimestamp());
    }

    bool TCookieMixin::IsValidRegexp(const TString& value) {
        return RE2::FullMatch(value, COOKIE_REGEXP);
    }

    TString TCookieMixin::GenerateRandomTimestamp(ui64 maxTimestamp) {
        if (maxTimestamp < MIN_TIMESTAMP) {
            ythrow yexception() << "maxTimestamp " << maxTimestamp <<
            " must be greater than " << MIN_TIMESTAMP;
        }

        TString timestamp = ToString<ui64>(
            MIN_TIMESTAMP + RandomNumber<ui64>(maxTimestamp - MIN_TIMESTAMP)
        );
        while (timestamp.size() < 10) {
            timestamp.prepend('0');
        }

        return timestamp;
    }

    TString TCookieMixin::GenerateRandomTimestamp() {
        return GenerateRandomTimestamp(GetMaxTimestamp() - 1000);
    }

    TString TCookieMixin::Next() {
        TString random_ts = GenerateRandomTimestamp(FIXED_MAX_TIMESTAMP);
        TString random_id(NextDigest());
        if (random_id[0] == '0') {
            random_id[0] = '1';
        }
        while (random_id.size() <  18 - random_ts.size()) {
            // allow zeros in random string
            random_id.append(NextDigest());
        }
        return random_id + random_ts;
    }

    bool TYandexuid::Validate(const TString& value) {
        return IsUint64(value) &&
               IsValidTimestamp(GetTimestamp(value)) &&
               IsValidRegexp(value);
    }

    bool TYandexuid::Validate(const ui64 value) {
        return IsValidTimestamp(GetTimestamp(value)) &&
           (value >= (10'000'000'000'000'000ull + MIN_TIMESTAMP));  // l17 with ts
            // (value <= 9'999'999'999'999'999'999ull);             // l20
    }

    bool TYandexuid::Significate(const ui64 value) {
        return Validate(value) && (value != 0);
    }

    ui64 TYandexuid::GetTimestamp() const {
        return GetTimestamp(Original);
    }

    TString TYandexuid::DoNormalize() const {
        return ToString<ui64>(FromString<ui64>(Original));
    }

    NCrypta::NIdentifiersProto::TGenericID TYandexuid::ToProto() const {
        auto proto = TIdentifier::ToProto();
        if (!proto.HasRawValue()) {
            proto.MutableYandexuid()->SetValue(FromString<ui64>(Normalize()));
        }
        return proto;
    }

    TString TYandexuid::FromProto(const NCrypta::NIdentifiersProto::TGenericID& proto) {
        return proto.HasRawValue() ? proto.GetRawValue() : ToString(proto.GetYandexuid().GetValue());
    }
}
