#include "ysclid.h"

#include <contrib/libs/re2/re2/re2.h>
#include <util/datetime/base.h>
#include <util/random/random.h>

namespace NIdentifiers {
    bool TYSClid::Validate(const TString& value) {
        static const re2::RE2 ysclidRegExp(R"(^[a-z0-9]{8,11}$)");
        static const re2::RE2 ysclidRegExpNew(R"(^[a-z0-9]{8,11}[0-9]{1,9}$)");

        return RE2::FullMatch(value, ysclidRegExp) || RE2::FullMatch(value, ysclidRegExpNew);
    }

    bool TYSClid::Significate(const TString& value) {
        const static ui64 significateTimestamp{1633035600}; // 2021-10-01

        return Validate(value) && GetTimestamp(value) > significateTimestamp;
    }

    TString TYSClid::Next() {
        return UintToStringWithBase(TInstant::Now().MilliSeconds(), 36)
            + UintToStringWithBase(RandomNumber<ui64>(36 * 36), 36)     // https://st.yandex-team.ru/SERP-139197#611e063efb7ddb5e8cae9ee5
            + UintToStringWithBase(RandomNumber<ui64>(1000000000), 10); // https://a.yandex-team.ru/review/2559679/details#file-0-180541627:R224
    }

    NCrypta::NIdentifiersProto::TGenericID TYSClid::ToProto() const {
        auto proto = TIdentifier::ToProto();
        if (!proto.HasRawValue()) {
            proto.MutableYSClid()->SetValue(Normalize());
        }
        return proto;
    }

    TString TYSClid::FromProto(const NCrypta::NIdentifiersProto::TGenericID& proto) {
        return proto.HasRawValue() ? proto.GetRawValue() : proto.GetYSClid().GetValue();
    }

    ui64 TYSClid::GetTimestamp() const {
        return TYSClid::GetTimestamp(Original);
    }

    ui64 TYSClid::GetTimestamp(const TString& value) {
        Y_ENSURE(TYSClid::Validate(value), "Timestamp can be calculated only from valid ysclid");
        return UintFromStringWithBase(value.substr(0, 8), 36) / 1000;
    }
}
