#include "identifier.h"
#include "likelihood.h"

#include <util/generic/algorithm.h>
#include <util/generic/yexception.h>

namespace NIdentifiers {
    TIdentifier::TIdentifier(const TStringBuf value, const EIdType& type)
        : Original(value)
        , Type(type)
    {
    }

    TString TIdentifier::Value() const {
        return Original;
    }

    TString TIdentifier::Normalize() const {
        if (!Normalized) {
            if (IsValid()) {
                Normalized = DoNormalize();
            } else {
                // fallback for invalid identifiers
                Normalized = Original;
            }
        }
        return Normalized.GetRef();
    }

    TString TIdentifier::StrictNormalize() const {
        if (!Normalized) {
            if (IsValid()) {
                Normalized = DoStrictNormalize();
            } else {
                // fallback for invalid identifiers
                Normalized = Original;
            }
        }
        return Normalized.GetRef();
    }

    bool TIdentifier::IsValid() const {
        if (!Valid) {
            Valid = (Original.length() <= MAX_IDENTIFIER_LENGTH) && Validate();
        }
        return Valid.GetRef();
    }

    bool TIdentifier::IsSignificant() const {
        if (!Significant) {
            Significant = Significate();
        }
        return Significant.GetRef();
    }

    size_t TIdentifier::Hash() const {
        return (THash<TString>()(Normalize())) ^ (THash<EIdType>()(Type));
    }

    TString TIdentifier::Next() {
        ythrow yexception() << "Non implemented for base class TIdentifier";
    }

    TString TIdentifier::GetMd5() const {
        if (!Md5hash) {
            Md5hash = ComputeMd5Hash();
        }
        return Md5hash.GetRef();
    }

    TString TIdentifier::GetSha256() const {
        if (!Sha256hash) {
            Sha256hash = ComputeSha256Hash();
        }
        return Sha256hash.GetRef();
    }

    ui64 TIdentifier::GetHalf() const {
        return ::ComputeYabsHash(Normalize());
    }

    TString TIdentifier::GetClearRepresentation() const {
        if (!ClearRepresentation) {
            ClearRepresentation = MakeClearRepresentation();
        }
        return ClearRepresentation.GetRef();
    }

    bool TIdentifier::operator==(const TIdentifier& other) const {
        // todo: does we want to verify Normalized on valid ids?
        return Type == other.Type && Normalize() == other.Normalize();
    }

    bool TIdentifier::HasLikelihood() const {
        return false;
    }

    double TIdentifier::GetFrequencySpread() {
        if (!HasLikelihood()) {
            ythrow yexception() << "no likelihood for this identifier";
        }
        return 0;
        // return NLikelihoodFeatures::TLikelihoodFeatures::Repetitions(GetClearRepresentation());
    }

    double TIdentifier::GetAlternating() {
        if (!HasLikelihood()) {
            ythrow yexception() << "no likelihood for this identifier";
        }
        return 0;
        // return NLikelihoodFeatures::TLikelihoodFeatures::Repetitions(GetClearRepresentation());
    }

    double TIdentifier::GetPeriodicity() {
        if (!HasLikelihood()) {
            ythrow yexception() << "no likelihood for this identifier";
        }
        return 0;
        // return NLikelihoodFeatures::TLikelihoodFeatures::Repetitions(GetClearRepresentation());
    }

    double TIdentifier::GetRepetitions() {
        if (!HasLikelihood()) {
            ythrow yexception() << "no likelihood for this identifier";
        }
        return 0;
        // return NLikelihoodFeatures::TLikelihoodFeatures::Repetitions(GetClearRepresentation());
    }

    EIdType TIdentifier::GetType() const {
        return Type;
    }

    TString TIdentifier::GetTypeString() const {
        return NCrypta::NIdentifiersProto::NIdType::EIdType_descriptor()
            ->FindValueByNumber(static_cast<i32>(Type))
            ->options()
            .GetExtension(NCrypta::NIdentifiersProto::Name);
    }

    NCrypta::NIdentifiersProto::TGenericID TIdentifier::ToProto() const {
        auto proto = NCrypta::NIdentifiersProto::TGenericID();
        proto.SetType(Type);
        if (!IsValid()) {
            proto.SetRawValue(Original);
        }
        return proto;
    }

    bool TIdentifier::Validate() const {
        return false;
    }

    bool TIdentifier::Significate() const {
        return true;
    }

    TString TIdentifier::DoNormalize() const {
        return Original;
    }

    TString TIdentifier::DoStrictNormalize() const {
        // in all common cases Strict and NonStrict are equal
        return DoNormalize();
    }

    TString TIdentifier::ComputeMd5Hash() const {
        return ::ComputeMd5Hash(Normalize());
    }

    TString TIdentifier::ComputeSha256Hash() const {
        return ::ComputeSha256Hash(Normalize());
    }

    TString TIdentifier::MakeClearRepresentation() const {
        TString value = Value();
        SubstGlobal(value, "-", "", 0);
        return value;
    }
}
