#pragma once

#include "blacklist.h"
#include "utils.h"

#include <crypta/lib/proto/identifiers/identifiers.pb.h>
#include <crypta/lib/proto/identifiers/id_type.pb.h>

#include <util/string/cast.h>
#include <util/generic/maybe.h>
#include <util/generic/string.h>
#include <util/generic/strbuf.h>
#include <util/string/subst.h>

using EIdType = NCrypta::NIdentifiersProto::NIdType::EIdType;

namespace NIdentifiers {

    class TIdentifier {
    public:
        explicit TIdentifier(const TStringBuf, const EIdType&);

        virtual ~TIdentifier() = default;

        TString Value() const;
        TString Normalize() const;
        TString StrictNormalize() const;
        bool IsValid() const;
        bool IsSignificant() const;
        size_t Hash() const;

        static TString Next();
        virtual TString GetNext() const = 0;

        TString GetMd5() const;
        TString GetSha256() const;
        ui64 GetHalf() const;
        TString GetClearRepresentation() const;

        virtual bool operator==(const TIdentifier&) const;

        virtual bool HasLikelihood() const;

        virtual double GetFrequencySpread();
        virtual double GetAlternating();
        virtual double GetPeriodicity();
        virtual double GetRepetitions();

        EIdType GetType() const;
        TString GetTypeString() const;

        virtual NCrypta::NIdentifiersProto::TGenericID ToProto() const;

        explicit operator ui64() const {
            try {
                return FromString<ui64>(Normalize());
            } catch (...) {
                return 0;
            }
        }

        explicit operator NCrypta::NIdentifiersProto::UInt256() const {
            try {
                return HexToUInt256(Normalize(), true);
            } catch (...) {
                return NCrypta::NIdentifiersProto::UInt256();
            }
        }

        virtual bool Validate() const;
        virtual bool Significate() const;
        virtual TString DoNormalize() const;
        virtual TString DoStrictNormalize() const;
        virtual TString ComputeMd5Hash() const;
        virtual TString ComputeSha256Hash() const;
        virtual TString MakeClearRepresentation() const;

    private:
        mutable TMaybe<TString> Normalized;
        mutable TMaybe<TString> Md5hash;
        mutable TMaybe<TString> Sha256hash;
        mutable TMaybe<TString> ClearRepresentation; // for likelihood funcs calculation
        mutable TMaybe<bool> Valid;
        mutable TMaybe<bool> Significant;

    protected:
        TString Original;
        EIdType Type;
    };

    template <class Identifier, EIdType idType>
    class TTypedIdentifier : public TIdentifier {
    public:
        explicit TTypedIdentifier(const TStringBuf identifierValue)
            : TIdentifier(identifierValue, idType) {
        }
        explicit TTypedIdentifier(const NCrypta::NIdentifiersProto::TGenericID& proto)
            : TIdentifier(Identifier::FromProto(proto), idType) {
        }

        virtual ~TTypedIdentifier() = default;

        virtual TString GetNext() const override {
            return Identifier::Next();
        }

        virtual bool Validate() const override {
            return Identifier::Validate(Value());
        }

        virtual bool Significate() const override {
            return Identifier::Significate(Value());
        }

        static bool Significate(const TString& value) {
            return Identifier::Validate(value)
                   && (Identifier::CountSignificant(value) > Identifier::MIN_SIGNIFICANT_COUNT)
                   // TODO(@zheglov) add static normalize interface and use it like Identifier::Normalize(value)
                   && !NBlackList::GetBlackList<Identifier>().contains(value)
                   ;
        }

        static const size_t MIN_SIGNIFICANT_COUNT{0u};
        static size_t CountSignificant(const TString& value) {
            return ::CountSignificant(value);
        }
    protected:
        explicit TTypedIdentifier(const TString& identifierValue, const EIdType& idTypeValue)
            : TIdentifier(identifierValue, idTypeValue) {
        }
    };
}

// define custom hash function for identifiers
template <>
struct THash<NIdentifiers::TIdentifier> {
    size_t operator()(const NIdentifiers::TIdentifier& identifier) const {
        return identifier.Hash();
    }
};
