#pragma once

#include <util/generic/algorithm.h>
#include <util/generic/map.h>
#include <util/generic/set.h>
#include <util/generic/string.h>
#include <util/generic/vector.h>
#include <util/generic/xrange.h>

namespace NLikelihoodFeatures {
    class TLikelihoodFeatures {
    public:
        explicit TLikelihoodFeatures() {};

        virtual ~TLikelihoodFeatures() = default;

        static double FrequencySpread(const TString &identifier);

        static double Alternating(const TString& identifier);

        static double Periodicity(const TString& identifier);

        static double Repetitions(const TString &identifier);

    protected:
        static double Variance(const TVector<int>& values, int amount) {
            double expected = 0;
            double squareExpected = 0;

            for (int val : values) {
                expected += val * (1.0 * val / amount);
                squareExpected += val * val * (1.0 * val / amount);
            }

            double variance = squareExpected - expected * expected;
            return variance;
        }

        static TVector<int> Frequencies(const TString& identifier) {
            TSet<int> symbols;
            for (TChar symbol : identifier) {
                symbols.insert(symbol);
            }

            TVector<int> freqs;
            for (TChar symbol : symbols) {
                freqs.push_back(Count(identifier, symbol));
            }
            return freqs;
        }

        static TVector<int> Signs(const TString& identifier) {
            TVector<int> signs;
            for (size_t i = 0; i < identifier.size() - 1; i++) {
                signs.push_back(Signum(int(identifier[i+1]) - int(identifier[i])));
            }
            return signs;
        }

        static TVector<TString> Periods(const TString& identifier) {
            TVector <TString> periods;
            for (size_t j = 1; j <= identifier.size() / 2; j++) {
                for (size_t i = 0; i < j; i++) {
                    TString cur_seq = "";
                    for (auto t : xrange(i, identifier.size(), j))
                        cur_seq += identifier[t];
                    periods.push_back(cur_seq);
                }
            }
            return periods;
        }

        static TMap<TString, int> SubstringsRepetitions(const TString& identifier) {
            TMap<TString, int> substr_freqs;
            for (size_t s_len = 2; s_len <= identifier.size(); s_len++) {
                for (size_t i = 0; i <= identifier.size() - s_len; i++) {
                    TString substr = identifier.substr(i, s_len);
                    substr_freqs.insert(std::pair<TString, int>(substr, CountSubstrFreq(identifier, substr)));
                }
            }
            return substr_freqs;
        }

    private:
        template <typename T> static int Signum(T val) {
            return (T(0) < val) - (val < T(0));
        }

        static int CountSubstrFreq(const TString& original, const TString& pattern) {
            size_t M = pattern.length();
            size_t N = original.length();
            int res = 0;
            for (size_t i = 0; i <= N - M; i++) {
                size_t j;
                for (j = 0; j < M; j++)
                    if (original[i+j] != pattern[j])
                        break;
                if (j == M) {
                    res++;
                }
            }
            return res;
        }
    };
}
