#include "identifier.h"
#include "utils.h"

#include <library/cpp/digest/md5/md5.h>
#include <contrib/libs/openssl/include/openssl/sha.h>

#include <util/generic/algorithm.h>
#include <util/generic/hash_set.h>
#include <util/generic/string.h>
#include <util/random/random.h>
#include <util/string/cast.h>
#include <util/string/hex.h>
#include <util/string/reverse.h>
#include <util/string/subst.h>
#include <util/system/types.h>

#include <crypta/lib/native/identifiers/lib/id_types/all.h>

#include <algorithm>

TString NextUInt() {
    return ToString<ui64>(RandomNumber<ui64>());
}

TString NextNonZeroUInt() {
    TString result;

    while((result = NextUInt()) == "0");

    return result;
}

char NextDigest() {
    return DIGEST[RandomNumber<ui32>(DIGEST.size())];
}

char NextHexDigest() {
    return HEX_DIGEST[RandomNumber<ui32>(HEX_DIGEST.size())];
}

char NextAlpha() {
    return ALPHABET[RandomNumber<ui32>(ALPHABET.size())];
}

char NextAlphaNumeric() {
    return ALPHA_NUMERIC[RandomNumber<ui32>(ALPHA_NUMERIC.size())];
}

char NextAlphaNumeric(const TString& extended) {
    const TString alpha_numeric_extended = ALPHA_NUMERIC + extended;
    return alpha_numeric_extended[RandomNumber<ui32>(alpha_numeric_extended.size())];
}

bool IsAllDigits(const TString& value) {
    if (value == "") {
        return false;
    }
    for (const auto& chr : value) {
        if (!isdigit(chr)) {
            return false;
        }
    }
    return true;
}

bool IsHexadecimal(const TString& orig) {
    TString value = to_lower(orig);
    SubstGlobal(value, "-", "", 0);
    return RE2::FullMatch(value, HEX_DIGEST_REGEXP);
}

bool IsUint(const TString& value, size_t max_len, const char* max_digits) {
    return (
        IsAllDigits(value) && (
            (value.length() < max_len) || (
                (value.length() == max_len) &&
                (value <= max_digits)
            )
        )
    );
}

bool IsUint32(const TString& value) {
    return IsUint(value, 10, "4294967295");  // (1 << 32) - 1
}

bool IsUint64(const TString& value) {
    return IsUint(value, 20, "18446744073709551615");  // (1 << 64) - 1
}

NCrypta::NIdentifiersProto::UInt128 HexToUInt128(const TString& orig, const bool /* autofill */) {
    TString value = to_lower(orig);
    SubstGlobal(value, "-", "", 0);

    if (value.size() != 32) {
        ythrow yexception() << "incorrect hexdigest size " << value.size();
    }

    ui64 lo = IntFromString<ui64, 16>(value.substr(16, 16));    // lowest bits       [16 .. 32)
    ui64 hi = IntFromString<ui64, 16>(value.substr(0, 16));     // higest bits [0 .. 16)

    auto proto = NCrypta::NIdentifiersProto::UInt128();

    proto.SetLo(lo);
    proto.SetHi(hi);

    return proto;
}

NCrypta::NIdentifiersProto::UInt256 HexToUInt256(const TString& orig, const bool autofill) {
    TString value = to_lower(orig);
    SubstGlobal(value, "-", "", 0);

    if (!autofill && value.size() != 64) {
        ythrow yexception() << "incorrect hexdigest size " << value.size();
    } else {
        while (value.size() < 64) {
            value.prepend("0");
        }
    }

    ui64 v0 = IntFromString<ui64, 16>(value.substr(48, 16));    // lowest bits                  [48 .. 64)
    ui64 v1 = IntFromString<ui64, 16>(value.substr(32, 16));    // middle bits            [32 .. 48)
    ui64 v2 = IntFromString<ui64, 16>(value.substr(16, 16));    // middle bits      [16 .. 32)
    ui64 v3 = IntFromString<ui64, 16>(value.substr(0, 16));     // higest bits [0 .. 16)

    auto proto = NCrypta::NIdentifiersProto::UInt256();

    proto.SetV0(v0);
    proto.SetV1(v1);
    proto.SetV2(v2);
    proto.SetV3(v3);

    return proto;
}

TString UIntToHex(ui64 value) {
    TString output = IntToString<16>(value);
    while (output.size() < 16) {
        output.prepend("0");
    }
    return to_lower(output);
}

TString UIntToHex(const NCrypta::NIdentifiersProto::UInt128& value) {
    return UIntToHex(value.GetHi()) + UIntToHex(value.GetLo());
}

TString UIntToHex(const NCrypta::NIdentifiersProto::UInt256& value) {
    return UIntToHex(value.GetV3()) + UIntToHex(value.GetV2()) + UIntToHex(value.GetV1()) + UIntToHex(value.GetV0());
}

TString HexDigestDashes(const TString& original) {
    TString value = original;
    value.insert(8, "-");
    value.insert(13, "-");
    value.insert(18, "-");
    value.insert(23, "-");
    return value;
}

TString ComputeMd5Hash(const TString& value) {
    return MD5::Calc(value);
}

TString ComputeSha256Hash(const TString& value) {
    SHA256_CTX sha;
    SHA256_Init(&sha);
    SHA256_Update(&sha, value.data(), value.size());
    unsigned char hash[SHA256_DIGEST_LENGTH];
    SHA256_Final(hash, &sha);

    return to_lower(HexEncode(hash, SHA256_DIGEST_LENGTH));
}

ui64 ComputeYabsHash(const TString& value) {
    ui8 buff[16];
    const auto md5string{MD5::Calc(value)};

    HexDecode(md5string.c_str(), md5string.size(), buff);

    ui64 res{0};
    for (int index{3}; index >= 0; --index) {
        res |= static_cast<ui64>(buff[0 + index] ^ buff[8 + index]) << ((3 - index) << 3);
        res |= static_cast<ui64>(buff[4 + index] ^ buff[12 + index]) << ((7 - index) << 3);
    }

    return res;
}

std::tuple<TString, TString> SplitEmail(const TString &email)  {
    size_t indexAt = email.find("@", 0);
    TString left = email.substr(0, indexAt);
    TString right = email.substr(indexAt + 1, email.length());

    return std::make_tuple(left, right);
}

bool CheckSignificant(const unsigned char ch) {
    return (
        // skip zeros
        (ch != '0')
        // skip all non alpha-numeric chars
        // such as: at, space, underscore, colum, etc...
        && static_cast<bool>(std::isalnum(ch))
    );
}

size_t CountSignificant(const TStringBuf value) {
    return std::count_if(value.cbegin(), value.cend(), CheckSignificant);
}

size_t CountSignificantUniq(const TStringBuf value) {
    THashSet<unsigned char> uniq;
    std::for_each(value.cbegin(), value.cend(), [&uniq](const unsigned char ch) {uniq.insert(ch);});
    return std::count_if(uniq.cbegin(), uniq.cend(), [](const unsigned char ch) { return std::isalnum(ch); });
}

TString UintToStringWithBase(ui64 value, int base) {
    Y_ENSURE(base > 1 && base < 37, "Base should be at least 2 and at most 36");

    if (value == 0) {
        return "0";
    }

    TString result;
    while (value) {
        auto next = value % base;

        Y_ENSURE(('0' <= next && '9' >= next) || ('a' >= next && 'z' >= next));

        result.push_back(next < 10 ? '0' + next : 'a' + next - 10);
        value /= base;
    }

    ReverseInPlace(result);
    return result;
}

ui64 UintFromStringWithBase(const TString& value, int base) {
    return std::stoull(value, nullptr, base);
}
