#include <iosfwd>

#include "helpers.h"

#include <crypta/lib/native/identifiers/lib/generic.h>
#include <crypta/lib/native/identifiers/lib/id_types/all.h>
#include <library/cpp/testing/unittest/registar.h>
#include <util/generic/hash.h>

template <class TypeIdentifier>
void IdentifierTest() {
    THashMap<TypeIdentifier, size_t> map{};
    TString prev{};
    for (size_t index{0}; index < 10; ++index) {
        const TString next{TypeIdentifier::Next()};
        UNIT_ASSERT(prev != next);

        const TypeIdentifier first{next};
        const TypeIdentifier second{next};
        UNIT_ASSERT_EQUAL(first, second);
        ++map[first];
        prev = next;
    }

    {
        size_t counter{0};
        for (const auto& [key, value] : map) {
            UNIT_ASSERT_EQUAL(value, 1);
            UNIT_ASSERT(key.IsValid());
            if (++counter < 5) {
                map[key] += counter;
            }
        }
        UNIT_ASSERT_EQUAL(counter, 10);
    }

    {
        size_t counter{0};
        for (const auto& [key, value] : map) {
            UNIT_ASSERT_EQUAL(value, ++counter < 5 ? counter + 1 : 1);
        }
        UNIT_ASSERT_EQUAL(counter, 10);
    }
}

template <class TypeIdentifier>
void GenericIdentifierTest() {
    using namespace NIdentifiers;
    THashMap<TGenericID, size_t> map{};
    const auto protoType{TypeIdentifier("").GetType()};
    const auto stringType{TypeIdentifier("").GetTypeString()};
    TString prev{};
    for (size_t index{0}; index < 10; ++index) {
        const TString next{TypeIdentifier::Next()};
        UNIT_ASSERT(prev != next);
        // proto and string type should create same object with hash ok
        const TGenericID protoBase{protoType, next};
        const TGenericID stringBase{stringType, next};
        // UNIT_ASSERT_EQUAL(protoBase, stringBase);
        ++map[protoBase];
        ++map[stringBase];
        prev = next;
    }

    {
        size_t counter{0};
        for (const auto& [key, value] : map) {
            UNIT_ASSERT_EQUAL(value, 2);
            UNIT_ASSERT(key.IsValid());
            if (++counter < 5) {
                map[key] += counter;
            }
        }
        UNIT_ASSERT_EQUAL(counter, 10);
    }

    {
        size_t counter{0};
        for (const auto& [key, value] : map) {
            UNIT_ASSERT_EQUAL(value, ++counter < 5 ? counter + 2 : 2);
        }
        UNIT_ASSERT_EQUAL(counter, 10);
    }
}

Y_UNIT_TEST_SUITE(TestHashIdentifier) {

    Y_UNIT_TEST(TEST_Hash_simple) {
        const NIdentifiers::TYandexuid yuid{"1234567891532376011"};
        const size_t simpleHash{THash<TString>()("1234567891532376011")
                                ^ THash<EIdType>()(yuid.GetType())};
        UNIT_ASSERT_EQUAL(simpleHash, 15623545991281213662ull);
        UNIT_ASSERT_EQUAL(THash<NIdentifiers::TIdentifier>()(yuid), simpleHash);
    }

    Y_UNIT_TEST(TEST_Hash_idfagaid_simple) {
        const NIdentifiers::TIdfaGaid first{"1234567891532376011"};
        const NIdentifiers::TIdfaGaid second{"1234567891532376011"};
        const NIdentifiers::TIdfaGaid third{"1234567891532376012"};
        const NIdentifiers::TIdfa fourth{"1234567891532376012"};
        UNIT_ASSERT_EQUAL(first, second);
        UNIT_ASSERT_UNEQUAL(first, third);
        UNIT_ASSERT_UNEQUAL(first, fourth);
        UNIT_ASSERT_UNEQUAL(third, fourth);

        UNIT_ASSERT_EQUAL(THash<NIdentifiers::TIdentifier>()(first), THash<NIdentifiers::TIdentifier>()(second));
        UNIT_ASSERT_UNEQUAL(THash<NIdentifiers::TIdentifier>()(first), THash<NIdentifiers::TIdentifier>()(third));
        UNIT_ASSERT_UNEQUAL(THash<NIdentifiers::TIdentifier>()(first), THash<NIdentifiers::TIdentifier>()(fourth));
        UNIT_ASSERT_UNEQUAL(THash<NIdentifiers::TIdentifier>()(third), THash<NIdentifiers::TIdentifier>()(fourth));
    }

#define TMP_TEST_MACRO(TIdType)                     \
    Y_UNIT_TEST(TEST_##TIdType##_Hash_direct) {     \
        using namespace NIdentifiers;               \
        IdentifierTest< TIdType >();                \
    }

    APPLY_TMP_MACRO_FOR_ALL_ID_TYPES()
#undef TMP_TEST_MACRO

    Y_UNIT_TEST(TEST_Hash_generic_simple) {
        using namespace NIdentifiers;

        const TGenericID yuid{"yandexuid", "1234567891532376011"};
        const size_t simpleHash{THash<TString>()("1234567891532376011") ^ THash<EIdType>()(yuid.GetType())};
        UNIT_ASSERT_EQUAL(simpleHash, 15623545991281213662ull);
        UNIT_ASSERT_EQUAL(THash<TGenericID>()(yuid), simpleHash);
    }

    Y_UNIT_TEST(TEST_Hash_generic_idfa_gaid) {
        using namespace NIdentifiers;

        const TGenericID first{"idfa", "1234567891532376011"};
        const TGenericID second{"idfa", "1234567891532376011"};
        const TGenericID third{"idfa", "1234567891532376012"};
        const TGenericID fourth{"gaid", "1234567891532376012"};

        UNIT_ASSERT_EQUAL(first, second);
        UNIT_ASSERT_UNEQUAL(first, third);
        UNIT_ASSERT_UNEQUAL(first, fourth);
        UNIT_ASSERT_UNEQUAL(third, fourth);

        UNIT_ASSERT_EQUAL(THash<TGenericID>()(first), THash<TGenericID>()(second));
        UNIT_ASSERT_UNEQUAL(THash<TGenericID>()(first), THash<TGenericID>()(third));
        UNIT_ASSERT_UNEQUAL(THash<TGenericID>()(first), THash<TGenericID>()(fourth));
        UNIT_ASSERT_UNEQUAL(THash<TGenericID>()(third), THash<TGenericID>()(fourth));
    }

    Y_UNIT_TEST(TEST_Hash_generic_and_concrete) {
        using namespace NIdentifiers;

        const TGenericID gIdfa{"idfa", "1234567891532376011"};
        const TGenericID gGaid{"gaid", "1234567891532376011"};
        const TIdfa cIdfa{"1234567891532376011"};
        const TGaid cGaid{"1234567891532376011"};

        // equal correct pairs
        UNIT_ASSERT_EQUAL(gIdfa, cIdfa);
        UNIT_ASSERT_EQUAL(gGaid, cGaid);
        UNIT_ASSERT_EQUAL(cIdfa, gIdfa);
        UNIT_ASSERT_EQUAL(cGaid, gGaid);

        // // unqeual cross pairs
        UNIT_ASSERT_UNEQUAL(gIdfa, cGaid);
        UNIT_ASSERT_UNEQUAL(gGaid, cIdfa);
        UNIT_ASSERT_UNEQUAL(cIdfa, gGaid);
        UNIT_ASSERT_UNEQUAL(cGaid, gIdfa);
        UNIT_ASSERT_UNEQUAL(cIdfa, cGaid);
        UNIT_ASSERT_UNEQUAL(gIdfa, gGaid);

        UNIT_ASSERT_EQUAL(THash<TGenericID>()(gIdfa), THash<TIdentifier>()(cIdfa));
        UNIT_ASSERT_EQUAL(THash<TGenericID>()(gGaid), THash<TIdentifier>()(cGaid));

        UNIT_ASSERT_UNEQUAL(THash<TGenericID>()(gIdfa), THash<TIdentifier>()(cGaid));
        UNIT_ASSERT_UNEQUAL(THash<TGenericID>()(gGaid), THash<TIdentifier>()(cIdfa));
        UNIT_ASSERT_UNEQUAL(THash<TIdentifier>()(cIdfa), THash<TIdentifier>()(cGaid));
        UNIT_ASSERT_UNEQUAL(THash<TGenericID>()(gIdfa), THash<TGenericID>()(gGaid));
    }

#define TMP_TEST_MACRO(TIdType)                     \
    Y_UNIT_TEST(TEST_##TIdType##_Hash_generic) {    \
        using namespace NIdentifiers;               \
        GenericIdentifierTest< TIdType >();         \
    }

    APPLY_TMP_MACRO_FOR_ALL_ID_TYPES()
#undef TMP_TEST_MACRO

    Y_UNIT_TEST(TEST_Hash_generic_multitype) {
        using namespace NIdentifiers;
        THashMap<TGenericID, size_t> map{};
        for (const auto& [stringType, protoType] : IDENTIFIERS_NAME_TO_TYPE) {
            const TGenericID protoBase{protoType};
            const TGenericID stringBase{stringType};

            // in all.h we have multi aliases for each type
            const size_t current{map[stringBase]};
            ++map[protoBase];
            ++map[stringBase];
            UNIT_ASSERT_EQUAL(map[stringBase], current + 2);
        }

        for (const auto& [id, count] : map) {
            UNIT_ASSERT(count > 1);
        }

        UNIT_ASSERT_EQUAL(map.size(), 47);
        UNIT_ASSERT_EQUAL(IDENTIFIERS_NAME_TO_TYPE.size(), 74);
    }

    Y_UNIT_TEST(TEST_Hash_half) {
        using namespace NIdentifiers;
        {
            const TGenericID gId{"gaid", "fc10dd733d50481c910974234ed69642"};
            const TGaid cId{"fc10dd733d50481c910974234ed69642"};
            UNIT_ASSERT_EQUAL(cId.GetHalf(), gId.GetHalf());
            UNIT_ASSERT_EQUAL(16576150013839068642ull, cId.GetHalf());
        }
        {
            const TGenericID gId{"idfa", "fc10dd733d50481c910974234ed69642"};
            const TIdfa cId{"fc10dd733d50481c910974234ed69642"};
            UNIT_ASSERT_EQUAL(cId.GetHalf(), gId.GetHalf());
            UNIT_ASSERT_EQUAL(10591752310015382839ull, cId.GetHalf());
        }
    }

    Y_UNIT_TEST(Test_Hash_Normalization) {
        using namespace NIdentifiers;

        const TGaid lowerCased{"fc10dd733d50481c910974234ed69642"};
        const TGaid upperCased{"FC10DD733D50481C910974234ED69642"};

        UNIT_ASSERT_EQUAL(lowerCased.Hash(), upperCased.Hash());
    }
}
