#include <iosfwd>

#include <crypta/lib/native/identifiers/lib/generic.h>
#include <crypta/lib/native/identifiers/lib/id_types/all.h>
#include <crypta/lib/native/identifiers/lib/likelihood.h>
#include <library/cpp/testing/unittest/registar.h>

static const double EPS = 0.000001;
static const double ABSENT_LIKELIHOOD = -100500;

// class for testing protected methods of class TIdentifer
class TIdentTest: public NIdentifiers::TIdentifier {
public:
    explicit TIdentTest(const TString& value)
        : NIdentifiers::TIdentifier(value, NCrypta::NIdentifiersProto::NIdType::DEFAULT) {
    }

    bool WrapIsUint32() {
        return IsUint32(Original);
    };

    bool WrapIsUint64() {
        return IsUint64(Original);
    };

    double GetFrequencySpread() override {
        return ABSENT_LIKELIHOOD;
    }

    double GetAlternating() override {
        return ABSENT_LIKELIHOOD;
    }

    double GetPeriodicity() override {
        return ABSENT_LIKELIHOOD;
    }

    double GetRepetitions() override {
        return ABSENT_LIKELIHOOD;
    }

    bool HasLikelihood() const override {
        return false;
    }

    TString GetNext() const override {
        return "";
    }

protected:
    bool Validate() const override {
        return true;
    }
};

// class for testing protected methods of class TYandexuid
class TYandexTest: public NIdentifiers::TYandexuid {
public:
    explicit TYandexTest(const TString& value)
        : NIdentifiers::TYandexuid(value)
    {
    }

    bool WrapIsValidTimestamp() {
        return IsValidTimestamp(GetTimestamp());
    };
};

class TEmailTest: public NIdentifiers::TEmail {
public:
    explicit TEmailTest(const TString& value)
        : NIdentifiers::TEmail(value)
    {
    }

    bool WrapIsUnnormalEmail() {
        return IsUnnormalEmail(Original);
    }

    bool WrapIsValidHeuristic() {
        return IsValidHeuristic(Original);
    }
};

class TLikelihoodTest : public NLikelihoodFeatures::TLikelihoodFeatures {
public:
    static double WrapVariance(const TVector<int>& values, int amount) {
        return Variance(values, amount);
    }

    static TVector<int> WrapFrequencies(const TString& identifier) {
        return Frequencies(identifier);
    }

    static TVector<int> WrapSigns(const TString& identifier) {
        return Signs(identifier);
    }

    static TVector<TString> WrapPeriods(const TString& identifier) {
        return Periods(identifier);
    }

    static TMap<TString, int> WrapSubstringsRepetitions(const TString& identifier) {
        return SubstringsRepetitions(identifier);
    }
};

template <class IdentifierType>
void TestGenericIdProto() {
    auto identifier = IdentifierType(IdentifierType::Next());
    auto generic_id = NIdentifiers::TGenericID(identifier.ToProto());
    {
        // construct from proto
        UNIT_ASSERT(generic_id.IsValid());
        UNIT_ASSERT_EQUAL(generic_id.GetType(), identifier.GetType());
        UNIT_ASSERT_EQUAL(generic_id.GetValue(), identifier.Value());
        UNIT_ASSERT_EQUAL(generic_id.Normalize(), identifier.Normalize());
    }
    {
        // dump to proto
        auto identifier_recovered = IdentifierType(generic_id.ToProto());
        UNIT_ASSERT(identifier_recovered.IsValid());
        UNIT_ASSERT_EQUAL(identifier_recovered.GetType(), identifier.GetType());
        UNIT_ASSERT_EQUAL(identifier_recovered.Value(), identifier.Value());
        UNIT_ASSERT_EQUAL(identifier_recovered.Normalize(), identifier.Normalize());
    }
}

Y_UNIT_TEST_SUITE(TUnitStatsTest) {
    Y_UNIT_TEST(Test_TIdentifier_ISUint32) {
        TIdentTest test1("83948uf43f");   // not correct, not the number
        TIdentTest test2("-39273264832"); // not correct, negative number
        TIdentTest test3("");             // not correct, empty
        TIdentTest test4("100260129");    // correct, length < 10
        TIdentTest test5("4294967294");   // correct, length == 10, value < MAX_UINT64
        TIdentTest test6("4294967295");   // correct, length == 10, value == MAX_UINT64
        TIdentTest test7("4294967296");   // not correct, length == 10, value > MAX_UINT64
        TIdentTest test8("42949672956");  // not correct, lenght > 10

        UNIT_ASSERT_EQUAL(test1.WrapIsUint32(), false);
        UNIT_ASSERT_EQUAL(test2.WrapIsUint32(), false);
        UNIT_ASSERT_EQUAL(test3.WrapIsUint32(), false);
        UNIT_ASSERT_EQUAL(test4.WrapIsUint32(), true);
        UNIT_ASSERT_EQUAL(test5.WrapIsUint32(), true);
        UNIT_ASSERT_EQUAL(test6.WrapIsUint32(), true);
        UNIT_ASSERT_EQUAL(test7.WrapIsUint32(), false);
        UNIT_ASSERT_EQUAL(test8.WrapIsUint32(), false);
    }

    Y_UNIT_TEST(Test_TIdentifier_ISUint64) {
        TIdentTest test1("83948uf43f");            // not correct, not the number
        TIdentTest test2("-39273264832");          // not correct, negative number
        TIdentTest test3("");                      // not correct, empty
        TIdentTest test4("1002601291526721902");   // correct, length < 20
        TIdentTest test5("18446744073709551614");  // correct, length == 20, value < MAX_UINT64
        TIdentTest test6("18446744073709551615");  // correct, length == 20, value == MAX_UINT64
        TIdentTest test7("18446744073709551615");  // correct, length == 20, value > MAX_UINT64
        TIdentTest test8("184467440737095516155"); // not correct, lenght > 20

        UNIT_ASSERT_EQUAL(test1.WrapIsUint64(), false);
        UNIT_ASSERT_EQUAL(test2.WrapIsUint64(), false);
        UNIT_ASSERT_EQUAL(test3.WrapIsUint64(), false);
        UNIT_ASSERT_EQUAL(test4.WrapIsUint64(), true);
        UNIT_ASSERT_EQUAL(test5.WrapIsUint64(), true);
        UNIT_ASSERT_EQUAL(test6.WrapIsUint64(), true);
        UNIT_ASSERT_EQUAL(test7.WrapIsUint64(), true);
        UNIT_ASSERT_EQUAL(test8.WrapIsUint64(), false);
    }

    Y_UNIT_TEST(Test_TYandexuid_IsValidTimestamp) {
        TYandexTest test1("900120000");    // correct
        TYandexTest test2("900126700");    // correct
        TYandexTest test3("900126790");    // correct
        TYandexTest test4("1234");         // not correct, small value
        TYandexTest test5("900000000");    // correct, value == MIN
        TYandexTest test6("000900000000"); // correct, value == MIN with front 0
        TYandexTest test7("1277653450");   // correct
        TYandexTest test8("4433327050");   // not correct, 2020 year
        TYandexTest test9("1563508333");   // ncorrect, 2019

        UNIT_ASSERT_EQUAL(test1.WrapIsValidTimestamp(), true);
        UNIT_ASSERT_EQUAL(test2.WrapIsValidTimestamp(), true);
        UNIT_ASSERT_EQUAL(test3.WrapIsValidTimestamp(), true);
        UNIT_ASSERT_EQUAL(test4.WrapIsValidTimestamp(), false);
        UNIT_ASSERT_EQUAL(test5.WrapIsValidTimestamp(), true);
        UNIT_ASSERT_EQUAL(test6.WrapIsValidTimestamp(), true);
        UNIT_ASSERT_EQUAL(test7.WrapIsValidTimestamp(), true);
        UNIT_ASSERT_EQUAL(test8.WrapIsValidTimestamp(), false);
        UNIT_ASSERT_EQUAL(test9.WrapIsValidTimestamp(), true);
    }

    Y_UNIT_TEST(Test_TYandexuid_test_valid) {
        using namespace NIdentifiers;
        // simple ok
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("454813711337817544"), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("698401770942865766"), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("540584871475090252"), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("878751130946427673"), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("976892251516639816"), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("745541441095092719"), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("265414080960508690"), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("733033001086253878"), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("762862601352033664"), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("222497521310172071"), true);
        // extreme ok
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("222497520900000000"), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("11249751310172071"), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("10000001310172071"), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("10000000900000000"), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("2111249751310172071"), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("9999999990900000000"), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("9999999991599999999"), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("11222497521310172071"), true);  // long 20
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("10000000000900000000"), true);  // long 20
        // not ok
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("10000000000000000"), false);  // icookie valid
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("0"), false);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("222497520099999999"), false);  // past
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("222497521810172071"), false);  // future
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("497521510172071"), false);     // short
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("001497521510172071"), false);  // zeros
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("1510172071"), false);  // super short
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("10000000099999999"), false);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("1000000900000000"), false);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("101222497521310172071"), false);  // long 21
        UNIT_ASSERT_EQUAL(TYandexuid::Validate("100000000000900000000"), false);  // long 21
    }

    Y_UNIT_TEST(Test_IdNameToType_Yandexuid) {
        using namespace NIdentifiers;
        UNIT_ASSERT_EQUAL(IDENTIFIERS_NAME_TO_TYPE.at("yandexuid"), EIdType::YANDEXUID);
        UNIT_ASSERT_EQUAL(IDENTIFIERS_NAME_TO_TYPE.at("yuid"), EIdType::YANDEXUID);
        UNIT_ASSERT_EQUAL(IDENTIFIERS_NAME_TO_TYPE.at("yandex-uid"), EIdType::YANDEXUID);
        UNIT_ASSERT_EQUAL(IDENTIFIERS_NAME_TO_TYPE.at("new-generated-yandex-uid"), EIdType::YANDEXUID);
        UNIT_ASSERT_EQUAL(IDENTIFIERS_NAME_TO_TYPE.at("turbo-yandex-uid"), EIdType::YANDEXUID);
        UNIT_ASSERT_EQUAL(IDENTIFIERS_NAME_TO_TYPE.at("yandex-uid-cookie"), EIdType::YANDEXUID);
        UNIT_ASSERT_EQUAL(IDENTIFIERS_NAME_TO_TYPE.at("yandex-uid-identify-cookie-match"), EIdType::YANDEXUID);
        UNIT_ASSERT_EQUAL(IDENTIFIERS_NAME_TO_TYPE.at("yandex-uid-query"), EIdType::YANDEXUID);
    }

    Y_UNIT_TEST(Test_TYandexuid_test_valid_generic) {
        using namespace NIdentifiers;
        // simple ok
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "454813711337817544").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "698401770942865766").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "540584871475090252").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "878751130946427673").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "976892251516639816").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "745541441095092719").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "265414080960508690").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "733033001086253878").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "762862601352033664").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "222497521310172071").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("yandex-uid", "222497521310172071").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("new-generated-yandex-uid", "222497521310172071").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("turbo-yandex-uid", "222497521310172071").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("yandex-uid-cookie", "222497521310172071").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("yandex-uid-identify-cookie-match", "222497521310172071").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("yandex-uid-query", "222497521310172071").IsValid(), true);
        // extreme ok
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "222497520900000000").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "11249751310172071").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "10000001310172071").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "10000000900000000").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "2111249751310172071").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "9999999990900000000").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "9999999991599999999").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "11222497521310172071").IsValid(), true);  // long 20
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "10000000000900000000").IsValid(), true);  // long 20
        // not ok
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "10000000000000000").IsValid(), false);  // icookie valid
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "0").IsValid(), false);
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "222497520099999999").IsValid(), false);  // past
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "222497521810172071").IsValid(), false);  // future
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "497521510172071").IsValid(), false);     // short
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "001497521510172071").IsValid(), false);  // zeros
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "1510172071").IsValid(), false);  // super short
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "10000000099999999").IsValid(), false);
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "1000000900000000").IsValid(), false);
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "101222497521310172071").IsValid(), false);  // long 21
        UNIT_ASSERT_EQUAL(TGenericID("yandexuid", "100000000000900000000").IsValid(), false);  // long 21
    }

    Y_UNIT_TEST(Test_TYandexuid_test_valid_uint) {
        using namespace NIdentifiers;
        // simple ok
        UNIT_ASSERT_EQUAL(TYandexuid::Validate(454813711337817544ull), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate(698401770942865766ull), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate(540584871475090252ull), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate(878751130946427673ull), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate(976892251516639816ull), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate(745541441095092719ull), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate(265414080960508690ull), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate(733033001086253878ull), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate(762862601352033664ull), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate(222497521310172071ull), true);
        // extreme ok
        UNIT_ASSERT_EQUAL(TYandexuid::Validate(222497520900000000ull), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate(11249751310172071ull), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate(10000001310172071ull), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate(10000000900000000ull), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate(2111249751310172071ull), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate(9999999990900000000ull), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate(9999999991599999999ull), true);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate(11222497521310172071ull), true);  // long 20
        UNIT_ASSERT_EQUAL(TYandexuid::Validate(10000000000900000000ull), true);  // long 20
        // not ok
        UNIT_ASSERT_EQUAL(TYandexuid::Validate(10000000000000000ull), false);  // icookie valid
        UNIT_ASSERT_EQUAL(TYandexuid::Validate(0ull), false);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate(222497520099999999ull), false);  // past
        UNIT_ASSERT_EQUAL(TYandexuid::Validate(222497521810172071ull), false);  // future
        UNIT_ASSERT_EQUAL(TYandexuid::Validate(497521510172071ull), false);     // short
        UNIT_ASSERT_EQUAL(TYandexuid::Validate(1510172071), false);  // super short
        UNIT_ASSERT_EQUAL(TYandexuid::Validate(10000000099999999ull), false);
        UNIT_ASSERT_EQUAL(TYandexuid::Validate(1000000900000000ull), false);
        // UNIT_ASSERT_EQUAL(TYandexuid::Validate(101222497521310172071ull), false);  // long 21
        // UNIT_ASSERT_EQUAL(TYandexuid::Validate(100000000000900000000ull), false);  // long 21
    }

    Y_UNIT_TEST(Test_TIcookie_test_valid) {
        using namespace NIdentifiers;
        // simple ok
        UNIT_ASSERT_EQUAL(TIcookie::Validate("454813711337817544"), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate("698401770942865766"), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate("540584871475090252"), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate("878751130946427673"), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate("976892251516639816"), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate("745541441095092719"), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate("265414080960508690"), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate("733033001086253878"), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate("762862601352033664"), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate("222497521310172071"), true);
        // extreme ok
        UNIT_ASSERT_EQUAL(TIcookie::Validate("222497520900000000"), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate("11249751310172071"), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate("10000001310172071"), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate("10000000900000000"), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate("2111249751310172071"), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate("9999999990900000000"), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate("9999999991599999999"), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate("10000000000000000"), true);  // icookie valid
        UNIT_ASSERT_EQUAL(TIcookie::Validate("222497520099999999"), true);  // past
        UNIT_ASSERT_EQUAL(TIcookie::Validate("10000000099999999"), true);
        // not ok
        UNIT_ASSERT_EQUAL(TIcookie::Validate("0"), false);
        UNIT_ASSERT_EQUAL(TIcookie::Validate("222497521810172071"), false);  // future
        UNIT_ASSERT_EQUAL(TIcookie::Validate("497521510172071"), false);     // short
        UNIT_ASSERT_EQUAL(TIcookie::Validate("001497521510172071"), false);  // zeros
        UNIT_ASSERT_EQUAL(TIcookie::Validate("1510172071"), false);  // super short
        UNIT_ASSERT_EQUAL(TIcookie::Validate("1000000900000000"), false);
        UNIT_ASSERT_EQUAL(TIcookie::Validate("11222497521310172071"), false);  // long 20
        UNIT_ASSERT_EQUAL(TIcookie::Validate("10000000000900000000"), false);  // long 20
        UNIT_ASSERT_EQUAL(TIcookie::Validate("101222497521310172071"), false);  // long 21
        UNIT_ASSERT_EQUAL(TIcookie::Validate("100000000000900000000"), false);  // long 21
    }

    Y_UNIT_TEST(Test_TIcookie_test_valid_generic) {
        using namespace NIdentifiers;
        // simple ok
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "454813711337817544").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "698401770942865766").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "540584871475090252").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "878751130946427673").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "976892251516639816").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "745541441095092719").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "265414080960508690").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "733033001086253878").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "762862601352033664").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "222497521310172071").IsValid(), true);
        // extreme ok
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "222497520900000000").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "11249751310172071").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "10000001310172071").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "10000000900000000").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "2111249751310172071").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "9999999990900000000").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "9999999991599999999").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "10000000000000000").IsValid(), true);  // icookie valid
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "222497520099999999").IsValid(), true);  // past
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "10000000099999999").IsValid(), true);
        // not ok
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "0").IsValid(), false);
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "222497521810172071").IsValid(), false);  // future
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "497521510172071").IsValid(), false);     // short
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "001497521510172071").IsValid(), false);  // zeros
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "1510172071").IsValid(), false);  // super short
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "1000000900000000").IsValid(), false);
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "11222497521310172071").IsValid(), false);  // long 20
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "10000000000900000000").IsValid(), false);  // long 20
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "101222497521310172071").IsValid(), false);  // long 21
        UNIT_ASSERT_EQUAL(TGenericID("icookie", "100000000000900000000").IsValid(), false);  // long 21
    }

    Y_UNIT_TEST(Test_TIcookie_test_valid_uint) {
        using namespace NIdentifiers;
        // simple ok
        UNIT_ASSERT_EQUAL(TIcookie::Validate(454813711337817544ull), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate(698401770942865766ull), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate(540584871475090252ull), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate(878751130946427673ull), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate(976892251516639816ull), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate(745541441095092719ull), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate(265414080960508690ull), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate(733033001086253878ull), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate(762862601352033664ull), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate(222497521310172071ull), true);
        // extreme ok
        UNIT_ASSERT_EQUAL(TIcookie::Validate(222497520900000000ull), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate(11249751310172071ull), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate(10000001310172071ull), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate(10000000900000000ull), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate(2111249751310172071ull), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate(9999999990900000000ull), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate(9999999991599999999ull), true);
        UNIT_ASSERT_EQUAL(TIcookie::Validate(10000000000000000ull), true);  // icookie valid
        UNIT_ASSERT_EQUAL(TIcookie::Validate(222497520099999999ull), true);  // past
        UNIT_ASSERT_EQUAL(TIcookie::Validate(10000000099999999ull), true);
        // not ok
        UNIT_ASSERT_EQUAL(TIcookie::Validate(0ull), false);
        UNIT_ASSERT_EQUAL(TIcookie::Validate(222497521810172071ull), false);  // future
        UNIT_ASSERT_EQUAL(TIcookie::Validate(497521510172071ull), false);     // short
        UNIT_ASSERT_EQUAL(TIcookie::Validate(1510172071), false);  // super short
        UNIT_ASSERT_EQUAL(TIcookie::Validate(1000000900000000ull), false);
        UNIT_ASSERT_EQUAL(TIcookie::Validate(11222497521310172071ull), false);  // long 20
        UNIT_ASSERT_EQUAL(TIcookie::Validate(10000000000900000000ull), false);  // long 20
        // UNIT_ASSERT_EQUAL(TIcookie::Validate(101222497521310172071ull), false);  // long 21
        // UNIT_ASSERT_EQUAL(TIcookie::Validate(100000000000900000000ull), false);  // long 21
    }

    Y_UNIT_TEST(Test_TDuid_test_valid) {
        using namespace NIdentifiers;
        // simple ok
        UNIT_ASSERT_EQUAL(TDuid::Validate("133781754445481371"), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate("104286576669840177"), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate("147509025254058487"), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate("104642767387875113"), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate("151663981697689225"), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate("109509271974554144"), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate("106050869026541408"), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate("108625387873303300"), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate("135203366476286260"), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate("131017207122249752"), true);
        UNIT_ASSERT_EQUAL(TDuid::Significate("15000000001234567890"), true);
        // extreme ok
        UNIT_ASSERT_EQUAL(TDuid::Validate("100000000022249752"), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate("13101720711124975"), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate("13101720711000000"), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate("10000000001000000"), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate("1310172071211124975"), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate("1000000000999999999"), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate("1599999999999999999"), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate("13101720711122249752"), true);  // long 20
        UNIT_ASSERT_EQUAL(TDuid::Validate("10000000001000000000"), true);  // long 20
        UNIT_ASSERT_EQUAL(TDuid::Validate("151017207100149752"), true);  // zeros
        UNIT_ASSERT_EQUAL(TDuid::Validate("100000000010000000"), true);
        UNIT_ASSERT_EQUAL(TDuid::Significate("15000000001234500"), true);
        // not ok
        UNIT_ASSERT_EQUAL(TDuid::Validate("09000000000000000"), false);  // icookie valid
        UNIT_ASSERT_EQUAL(TDuid::Validate("0"), false);
        UNIT_ASSERT_EQUAL(TDuid::Validate("009999999922249752"), false);  // past
        UNIT_ASSERT_EQUAL(TDuid::Validate("181017207122249752"), false);  // future
        UNIT_ASSERT_EQUAL(TDuid::Validate("151017207"), false);  // super short
        UNIT_ASSERT_EQUAL(TDuid::Validate("99999999991000000"), false);
        UNIT_ASSERT_EQUAL(TDuid::Validate("131017207110122249752"), false);  // long 21
        UNIT_ASSERT_EQUAL(TDuid::Validate("100000000010000000000"), false);  // long 21
        UNIT_ASSERT_EQUAL(TDuid::Significate("15000000001234"), false);
        // check for len
        UNIT_ASSERT_EQUAL(TDuid::Validate("10267356305"), false);
        UNIT_ASSERT_EQUAL(TDuid::Validate("111017826531"), false);
        UNIT_ASSERT_EQUAL(TDuid::Validate("1634430388375"), false);
        UNIT_ASSERT_EQUAL(TDuid::Validate("1634430388375388"), false);
        UNIT_ASSERT_EQUAL(TDuid::Validate("16344303883753883"), true);
    }

    Y_UNIT_TEST(Test_TDuid_test_valid_generic) {
        using namespace NIdentifiers;
        // simple ok
        UNIT_ASSERT_EQUAL(TGenericID("duid", "133781754445481371").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("duid", "104286576669840177").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("duid", "147509025254058487").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("duid", "104642767387875113").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("duid", "151663981697689225").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("duid", "109509271974554144").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("duid", "106050869026541408").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("duid", "108625387873303300").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("duid", "135203366476286260").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("duid", "131017207122249752").IsValid(), true);
        // extreme ok
        UNIT_ASSERT_EQUAL(TGenericID("duid", "100000000022249752").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("duid", "13101720711124975").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("duid", "13101720711000000").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("duid", "10000000001000000").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("duid", "1310172071211124975").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("duid", "1000000000999999999").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("duid", "1599999999999999999").IsValid(), true);
        UNIT_ASSERT_EQUAL(TGenericID("duid", "13101720711122249752").IsValid(), true);  // long 20
        UNIT_ASSERT_EQUAL(TGenericID("duid", "10000000001000000000").IsValid(), true);  // long 20
        UNIT_ASSERT_EQUAL(TGenericID("duid", "151017207100149752").IsValid(), true);  // zeros
        UNIT_ASSERT_EQUAL(TGenericID("duid", "100000000010000000").IsValid(), true);
        // not ok
        UNIT_ASSERT_EQUAL(TGenericID("duid", "09000000000000000").IsValid(), false);  // icookie valid
        UNIT_ASSERT_EQUAL(TGenericID("duid", "0").IsValid(), false);
        UNIT_ASSERT_EQUAL(TGenericID("duid", "009999999922249752").IsValid(), false);  // past
        UNIT_ASSERT_EQUAL(TGenericID("duid", "181017207122249752").IsValid(), false);  // future
        UNIT_ASSERT_EQUAL(TGenericID("duid", "151017207").IsValid(), false);  // super short
        UNIT_ASSERT_EQUAL(TGenericID("duid", "99999999991000000").IsValid(), false);
        UNIT_ASSERT_EQUAL(TGenericID("duid", "131017207110122249752").IsValid(), false);  // long 21
        UNIT_ASSERT_EQUAL(TGenericID("duid", "100000000010000000000").IsValid(), false);  // long 21
    }

    Y_UNIT_TEST(Test_TDuid_test_valid_uint) {
        using namespace NIdentifiers;
        // simple ok
        UNIT_ASSERT_EQUAL(TDuid::Validate(133781754445481371ull), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate(104286576669840177ull), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate(147509025254058487ull), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate(104642767387875113ull), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate(151663981697689225ull), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate(109509271974554144ull), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate(106050869026541408ull), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate(108625387873303300ull), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate(135203366476286260ull), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate(131017207122249752ull), true);
        // extreme ok
        UNIT_ASSERT_EQUAL(TDuid::Validate(100000000022249752ull), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate(13101720711124975ull), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate(13101720711000000ull), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate(10000000001000000ull), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate(1310172071211124975ull), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate(1000000000999999999ull), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate(1599999999999999999ull), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate(151017207100149752ull), true);  // zeros
        UNIT_ASSERT_EQUAL(TDuid::Validate(100000000010000000ull), true);
        UNIT_ASSERT_EQUAL(TDuid::Validate(10000000000000000000ull), true);  // long 20
        UNIT_ASSERT_EQUAL(TDuid::Validate(13101720711122249752ull), true);  // long 20
        // not ok
        UNIT_ASSERT_EQUAL(TDuid::Validate(9000000000000000ull), false);  // icookie valid
        UNIT_ASSERT_EQUAL(TDuid::Validate(0ull), false);
        UNIT_ASSERT_EQUAL(TDuid::Validate(9999999922249752ull), false);  // past
        UNIT_ASSERT_EQUAL(TDuid::Validate(181017207122249752ull), false);  // future
        UNIT_ASSERT_EQUAL(TDuid::Validate(151017207ull), false);  // super short
        UNIT_ASSERT_EQUAL(TDuid::Validate(99999999991000000ull), false);
        UNIT_ASSERT_EQUAL(TDuid::Validate(18446744073709551615ull), false); // max duid
        // UNIT_ASSERT_EQUAL(TDuid::Validate(131017207110122249752ull), false);  // long 21
        // UNIT_ASSERT_EQUAL(TDuid::Validate(100000000010000000000ull), false);  // long 21
    }

    Y_UNIT_TEST(Test_TYandexuid_GetTimestamp) {
        TString stringValue{"11188541530035229"};
        ui64 intValue{11188541530035229};

        UNIT_ASSERT_EQUAL(NIdentifiers::TYandexuid::GetTimestamp(stringValue), 1530035229);
        UNIT_ASSERT_EQUAL(NIdentifiers::TYandexuid::GetTimestamp(intValue), 1530035229);
    }

    Y_UNIT_TEST(Test_TYandexuid_IsValid) {
        NIdentifiers::TYandexuid test1("");                    // not correct, empty
        NIdentifiers::TYandexuid test2("0");                   // not correct, less than min
        NIdentifiers::TYandexuid test3("-1");                  // not correct, negative number
        NIdentifiers::TYandexuid test4("10113701529442803");   // correct
        NIdentifiers::TYandexuid test5("0011188541530035229"); // not correct, start with 0

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::YANDEXUID);

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), false);
        UNIT_ASSERT_EQUAL(test3.IsValid(), false);
        UNIT_ASSERT_EQUAL(test4.IsValid(), true);
        UNIT_ASSERT_EQUAL(test5.IsValid(), true);

        UNIT_ASSERT_EQUAL(test5.Normalize(), "11188541530035229");
        UNIT_ASSERT_EQUAL(test5.GetTimestamp(), 1530035229ull);
    }

    Y_UNIT_TEST(Test_TAndroidId) {
        NIdentifiers::TAndroidId test1("");
        NIdentifiers::TAndroidId test2("0");
        NIdentifiers::TAndroidId test3("628f59292a492110");
        NIdentifiers::TAndroidId test4("628F59292A492110");

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::ANDROID_ID);

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), false);
        UNIT_ASSERT_EQUAL(test3.IsValid(), true);
        UNIT_ASSERT_EQUAL(test4.IsValid(), true);
        UNIT_ASSERT_EQUAL(test3.Normalize(), "628f59292a492110");
        UNIT_ASSERT_EQUAL(test4.Normalize(), "628f59292a492110");
    }

    Y_UNIT_TEST(Test_TImei) {
        NIdentifiers::TImei test1("");
        NIdentifiers::TImei test2("00000000");
        NIdentifiers::TImei test3("A0000095318A17");
        NIdentifiers::TImei test4("99001307501073");
        NIdentifiers::TImei test5("099001307501073");
        NIdentifiers::TImei test6("356611086519745");

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::IMEI);

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), false);
        UNIT_ASSERT_EQUAL(test3.IsValid(), false);
        UNIT_ASSERT_EQUAL(test4.IsValid(), false);
        UNIT_ASSERT_EQUAL(test5.IsValid(), false);
        UNIT_ASSERT_EQUAL(test6.IsValid(), true);
    }

    Y_UNIT_TEST(Test_TPuid_IsValid) {
        NIdentifiers::TPuid test1("");                     // not correct, empty
        NIdentifiers::TPuid test2("0");                    // not correct, 0 is exception
        NIdentifiers::TPuid test3("12341234");             // correct
        NIdentifiers::TPuid test4("-123521345");           // not correct, negative number
        NIdentifiers::TPuid test5("123987dfdsf");          // not correct, not the number
        NIdentifiers::TPuid test6("4294967294");           // correct
        NIdentifiers::TPuid test7("4294967295");           // correct, value == MAX
        NIdentifiers::TPuid test8("4294967296");           // not correct
        NIdentifiers::TPuid test9("42949672234523967645"); // not correct
        NIdentifiers::TPuid test10("00123");
        NIdentifiers::TPuid test11("007007007");           // not correct, debug value

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::PUID);

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), false);
        UNIT_ASSERT_EQUAL(test3.IsValid(), true);
        UNIT_ASSERT_EQUAL(test4.IsValid(), false);
        UNIT_ASSERT_EQUAL(test5.IsValid(), false);
        UNIT_ASSERT_EQUAL(test6.IsValid(), true);
        UNIT_ASSERT_EQUAL(test7.IsValid(), true);
        UNIT_ASSERT_EQUAL(test8.IsValid(), true);
        UNIT_ASSERT_EQUAL(test9.IsValid(), false);
        UNIT_ASSERT_EQUAL(test11.IsValid(), false);

        UNIT_ASSERT_EQUAL(test10.Normalize(), "123");
    }

    Y_UNIT_TEST(Test_TDirectClientId_IsValid) {
        using namespace NIdentifiers;

        UNIT_ASSERT_EQUAL(TDirectClientId("").GetType(), EIdType::DIRECT_CLIENT_ID);

        UNIT_ASSERT_EQUAL(TDirectClientId("").IsValid(),            false);      // not correct, empty
        UNIT_ASSERT_EQUAL(TDirectClientId("0").IsValid(),           false);      // not correct, 0 is exception
        UNIT_ASSERT_EQUAL(TDirectClientId("12341234").IsValid(),    true);       // correct
        UNIT_ASSERT_EQUAL(TDirectClientId("-123521345").IsValid(),  false);      // not correct, negative number
        UNIT_ASSERT_EQUAL(TDirectClientId("123987dfdsf").IsValid(), false);      // not correct, not the number
        UNIT_ASSERT_EQUAL(TDirectClientId("9999999999").IsValid(),  true);       // correct, value == MAX
        UNIT_ASSERT_EQUAL(TDirectClientId("10000000000").IsValid(), false);      // not correct
    }

    Y_UNIT_TEST(Test_TIdfa_IsValid) {
        NIdentifiers::TIdfa test1("");                                            // not correct, empty
        NIdentifiers::TIdfa test2("F1D2AEE4-FDE7-4E18-A612-4EAB70DC2FCF");        // correct
        NIdentifiers::TIdfa test3("F1D2AEE4-fDE7-4E18-A612-4EAB70DC2FCF");        // correct, lower case
        NIdentifiers::TIdfa test4("F1D2AEE4-FDE7-4E18-A612-4EAB70DC2FCFS");       // not correct, bad length
        NIdentifiers::TIdfa test5("F1D2AEE4-FDE7-4E18-A612-4EA2FCFF1D2-0DC2FCF"); // not correct, bad length
        NIdentifiers::TIdfa test6("F1D2AEE4-0000-4E18-A612-4EAB70DC2FCF");        // correct

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::IDFA);

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), true);
        UNIT_ASSERT_EQUAL(test3.IsValid(), true);
        UNIT_ASSERT_EQUAL(test4.IsValid(), false);
        UNIT_ASSERT_EQUAL(test5.IsValid(), false);
        UNIT_ASSERT_EQUAL(test6.IsValid(), true);
    }

    Y_UNIT_TEST(Test_TIfv_IsValid) {
        NIdentifiers::TIfv test1("");                                            // not correct, empty
        NIdentifiers::TIfv test2("F1D2AEE4-FDE7-4E18-A612-4EAB70DC2FCF");        // correct
        NIdentifiers::TIfv test3("F1D2AEE4-fDE7-4E18-A612-4EAB70DC2FCF");        // correct, lower case
        NIdentifiers::TIfv test4("F1D2AEE4-FDE7-4E18-A612-4EAB70DC2FCFS");       // not correct, bad length
        NIdentifiers::TIfv test5("F1D2AEE4-FDE7-4E18-A612-4EA2FCFF1D2-0DC2FCF"); // not correct, bad length
        NIdentifiers::TIfv test6("F1D2AEE4-0000-4E18-A612-4EAB70DC2FCF");        // correct

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::IFV);

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), true);
        UNIT_ASSERT_EQUAL(test3.IsValid(), true);
        UNIT_ASSERT_EQUAL(test4.IsValid(), false);
        UNIT_ASSERT_EQUAL(test5.IsValid(), false);
        UNIT_ASSERT_EQUAL(test6.IsValid(), true);
    }

    Y_UNIT_TEST(Test_TGaid_IsValid) {
        NIdentifiers::TGaid test1("");                                            // not correct, empty
        NIdentifiers::TGaid test2("f1d2aee4-fde7-4e18-a612-4eab70dc2fcf");        // correct
        NIdentifiers::TGaid test3("f1d2aee4-fde7-4e18-a612-4eab70dc2Qcf");        // not correct, Q in record
        NIdentifiers::TGaid test4("f1d2aee4-fde7-4e18-a612-4eab70dc2fcfs");       // not correct, bad length
        NIdentifiers::TGaid test5("f1d2aee4-fde7-4e18-a612-4ea2fcff1d2-0dc2fcf"); // not correct, bad length
        NIdentifiers::TGaid test6("f1d2aee4-0000-4e18-a612-4eab70dc2fcf");        // correct

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::GAID);

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), true);
        UNIT_ASSERT_EQUAL(test3.IsValid(), false);
        UNIT_ASSERT_EQUAL(test4.IsValid(), false);
        UNIT_ASSERT_EQUAL(test5.IsValid(), false);
        UNIT_ASSERT_EQUAL(test6.IsValid(), true);
    }

    Y_UNIT_TEST(Test_TOaid_IsValid) {
        NIdentifiers::TOaid test1("");                                            // not correct, empty
        NIdentifiers::TOaid test2("f1d2aee4-fde7-4e18-a612-4eab70dc2fcf");        // correct
        NIdentifiers::TOaid test3("f1d2aee4-fde7-4e18-a612-4eab70dc2Qcf");        // not correct, Q in record
        NIdentifiers::TOaid test4("f1d2aee4-fde7-4e18-a612-4eab70dc2fcfs");       // not correct, bad length
        NIdentifiers::TOaid test5("f1d2aee4-fde7-4e18-a612-4ea2fcff1d2-0dc2fcf"); // not correct, bad length
        NIdentifiers::TOaid test6("f1d2aee4-0000-4e18-a612-4eab70dc2fcf");        // correct

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::OAID);

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), true);
        UNIT_ASSERT_EQUAL(test3.IsValid(), false);
        UNIT_ASSERT_EQUAL(test4.IsValid(), false);
        UNIT_ASSERT_EQUAL(test5.IsValid(), false);
        UNIT_ASSERT_EQUAL(test6.IsValid(), true);
    }

    Y_UNIT_TEST(Test_TMm_device_id_IsValid) {
        NIdentifiers::TMmDeviceId test1("");                                            // not correct, empty
        NIdentifiers::TMmDeviceId test2("F1D2AEE4-FDE7-4E18-A612-4EAB70DC2FCF");        // correct
        NIdentifiers::TMmDeviceId test3("f1d2aee4-fde7-4e18-a612-4eab70dc2Fcf");        // correct
        NIdentifiers::TMmDeviceId test4("f1d2aee4-fde7-4e18-a612-4eab70dc2fcfs");       // not correct, bad length
        NIdentifiers::TMmDeviceId test5("f1d2aee4-fde7-4e18-a612-4ea2fcff1d2-0dc2fcf"); // not correct, bad length
        NIdentifiers::TMmDeviceId test6("f1d2aee4-fde7-4e18-a612-4ed2-0d2fcf");         // not correct, bad length
        NIdentifiers::TMmDeviceId test7("f1d2aee4-0000-4e18-a612-4eab70dc2fcf");        // correct
        NIdentifiers::TMmDeviceId test8("00000000-1111-2222-3333-444444444444");        // correct
        NIdentifiers::TMmDeviceId test9("00000000111122223333444444444444");            // correct
        NIdentifiers::TMmDeviceId test10("a1a1a1a1a1a1a1a1a1a1AAAA");                   // correct
        NIdentifiers::TMmDeviceId test11("a1a1a1a1a1a1a1a1a1a1");                       // correct
        NIdentifiers::TMmDeviceId test12("a1a1a1a1a1a1a1a1a1a1a");                      // not correct
        NIdentifiers::TMmDeviceId test13("XKD2AEE4FDE74E18A6124EAB70DC2FCF");           // not correct

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::MM_DEVICE_ID);

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), true);
        UNIT_ASSERT_EQUAL(test3.IsValid(), true);
        UNIT_ASSERT_EQUAL(test4.IsValid(), false);
        UNIT_ASSERT_EQUAL(test5.IsValid(), false);
        UNIT_ASSERT_EQUAL(test6.IsValid(), false);
        UNIT_ASSERT_EQUAL(test7.IsValid(), true);
        UNIT_ASSERT_EQUAL(test8.IsValid(), true);
        UNIT_ASSERT_EQUAL(test9.IsValid(), true);
        UNIT_ASSERT_EQUAL(test10.IsValid(), true);
        UNIT_ASSERT_EQUAL(test11.IsValid(), true);
        UNIT_ASSERT_EQUAL(test12.IsValid(), false);
        UNIT_ASSERT_EQUAL(test13.IsValid(), false);
    }

    Y_UNIT_TEST(Test_TAlisa_device_id_IsValid) {
        NIdentifiers::TAlisaDeviceId test1("");                                            // not correct, empty
        NIdentifiers::TAlisaDeviceId test2("F1D2AEE4-FDE7-4E18-A612-4EAB70DC2FCF");        // correct
        NIdentifiers::TAlisaDeviceId test3("f1d2aee4-fde7-4e18-a612-4eab70dc2Fcf");        // correct
        NIdentifiers::TAlisaDeviceId test4("f1d2aee4-fde7-4e18-a612-4eab70dc2fcfs");       // not correct, bad length
        NIdentifiers::TAlisaDeviceId test5("f1d2aee4-fde7-4e18-a612-4ea2fcff1d2-0dc2fcf"); // not correct, bad length
        NIdentifiers::TAlisaDeviceId test6("f1d2aee4-fde7-4e18-a612-4ed2-0d2fcf");         // correct
        NIdentifiers::TAlisaDeviceId test7("f1d2aee4-0000-4e18-a612-4eab70dc2fcf");        // correct
        NIdentifiers::TAlisaDeviceId test8("00000000-1111-2222-3333-444444444444");        // correct
        NIdentifiers::TAlisaDeviceId test9("00000000111122223333444444444444");            // correct
        NIdentifiers::TAlisaDeviceId test10("a1a1a1a1a1a1a1a1a1a1AAAA");                   // correct
        NIdentifiers::TAlisaDeviceId test11("a1a1a1a1a1a1a1a1a1a1");                       // correct
        NIdentifiers::TAlisaDeviceId test12("a1a1a1a1a1a1a1a1a1a1a");                      // not correct
        NIdentifiers::TAlisaDeviceId test13("XKD2AEE4FDE74E18A6124EAB70DC2FCF");           // correct

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::ALISA_DEVICE_ID);

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), true);
        UNIT_ASSERT_EQUAL(test3.IsValid(), true);
        UNIT_ASSERT_EQUAL(test4.IsValid(), false);
        UNIT_ASSERT_EQUAL(test5.IsValid(), false);
        UNIT_ASSERT_EQUAL(test6.IsValid(), true);
        UNIT_ASSERT_EQUAL(test7.IsValid(), true);
        UNIT_ASSERT_EQUAL(test8.IsValid(), true);
        UNIT_ASSERT_EQUAL(test9.IsValid(), true);
        UNIT_ASSERT_EQUAL(test10.IsValid(), true);
        UNIT_ASSERT_EQUAL(test11.IsValid(), true);
        UNIT_ASSERT_EQUAL(test12.IsValid(), true);
        UNIT_ASSERT_EQUAL(test13.IsValid(), true);
    }

    Y_UNIT_TEST(Test_Blacklisting) {
        using namespace NIdentifiers;
        {
            TMmDeviceId id{"03000200040005000006000700080009"};
            UNIT_ASSERT_EQUAL(id.IsValid(), true);
            UNIT_ASSERT_EQUAL(id.IsSignificant(), false);
        }
        {
            TMmDeviceId id{"13000200040005000006000700080009"};
            UNIT_ASSERT_EQUAL(id.IsValid(), true);
            UNIT_ASSERT_EQUAL(id.IsSignificant(), true);
        }
        {
            TYandexuid id{"1134119271655967994"};
            UNIT_ASSERT_EQUAL(id.IsValid(), true);
            UNIT_ASSERT_EQUAL(id.IsSignificant(), false);
        }
        {
            TPuid id{"404744950"};
            UNIT_ASSERT_EQUAL(id.IsValid(), true);
            UNIT_ASSERT_EQUAL(id.IsSignificant(), false);
        }
    }

    Y_UNIT_TEST(Test_TMm_device_id_Normalize) {
        NIdentifiers::TMmDeviceId test1("F1D2AEE4-FDE7-4E18-A612-4EAB70DC2FCF");        // correct
        NIdentifiers::TMmDeviceId test2("f1d2aee4-fde7-4e18-a612-4eab70dc2Fcf");        // correct
        NIdentifiers::TMmDeviceId test3("f1d2aee4-0000-4e18-a612-4eab70dc2fcf");        // correct
        NIdentifiers::TMmDeviceId test4("00000000-1111-2222-3333-444444444444");        // correct
        NIdentifiers::TMmDeviceId test5("00000000111122223333444444444444");            // correct
        NIdentifiers::TMmDeviceId test6("a1a1a1a1a1a1a1a1a1a1AAAA");                    // correct
        NIdentifiers::TMmDeviceId test7("a1a1a1a1a1a1a1a1a1a1");                        // correct

        UNIT_ASSERT_EQUAL(test1.Normalize(), "F1D2AEE4-FDE7-4E18-A612-4EAB70DC2FCF");
        UNIT_ASSERT_EQUAL(test2.Normalize(), "f1d2aee4fde74e18a6124eab70dc2fcf");
        UNIT_ASSERT_EQUAL(test3.Normalize(), "f1d2aee400004e18a6124eab70dc2fcf");
        UNIT_ASSERT_EQUAL(test4.Normalize(), "00000000111122223333444444444444");
        UNIT_ASSERT_EQUAL(test5.Normalize(), "00000000111122223333444444444444");
        UNIT_ASSERT_EQUAL(test6.Normalize(), "00000000a1a1a1a1a1a1a1a1a1a1aaaa");
        UNIT_ASSERT_EQUAL(test7.Normalize(), "000000000000a1a1a1a1a1a1a1a1a1a1");
    }

    Y_UNIT_TEST(Test_TMm_device_id_StrictNormalize) {
        NIdentifiers::TMmDeviceId test1("F1D2AEE4-FDE7-4E18-A612-4EAB70DC2FCF");        // correct
        NIdentifiers::TMmDeviceId test2("f1d2aee4-fde7-4e18-a612-4eab70dc2Fcf");        // correct
        NIdentifiers::TMmDeviceId test3("f1d2aee4-0000-4e18-a612-4eab70dc2fcf");        // correct
        NIdentifiers::TMmDeviceId test4("00000000-1111-2222-3333-444444444444");        // correct
        NIdentifiers::TMmDeviceId test5("00000000111122223333444444444444");            // correct
        NIdentifiers::TMmDeviceId test6("a1a1a1a1a1a1a1a1a1a1AAAA");                    // correct
        NIdentifiers::TMmDeviceId test7("a1a1a1a1a1a1a1a1a1a1");                        // correct

        UNIT_ASSERT_EQUAL(test1.StrictNormalize(), "f1d2aee4fde74e18a6124eab70dc2fcf");
        UNIT_ASSERT_EQUAL(test2.StrictNormalize(), "f1d2aee4fde74e18a6124eab70dc2fcf");
        UNIT_ASSERT_EQUAL(test3.StrictNormalize(), "f1d2aee400004e18a6124eab70dc2fcf");
        UNIT_ASSERT_EQUAL(test4.StrictNormalize(), "00000000111122223333444444444444");
        UNIT_ASSERT_EQUAL(test5.StrictNormalize(), "00000000111122223333444444444444");
        UNIT_ASSERT_EQUAL(test6.StrictNormalize(), "00000000a1a1a1a1a1a1a1a1a1a1aaaa");
        UNIT_ASSERT_EQUAL(test7.StrictNormalize(), "000000000000a1a1a1a1a1a1a1a1a1a1");
    }

    Y_UNIT_TEST(Test_TMm_device_id_Generic_StrictNormalize) {
        NIdentifiers::TGenericID test1("mm_device_id", "F1D2AEE4-FDE7-4E18-A612-4EAB70DC2FCF");        // correct
        NIdentifiers::TGenericID test2("mm_device_id", "f1d2aee4-fde7-4e18-a612-4eab70dc2Fcf");        // correct
        NIdentifiers::TGenericID test3("mm_device_id", "f1d2aee4-0000-4e18-a612-4eab70dc2fcf");        // correct
        NIdentifiers::TGenericID test4("mm_device_id", "00000000-1111-2222-3333-444444444444");        // correct
        NIdentifiers::TGenericID test5("mm_device_id", "00000000111122223333444444444444");            // correct
        NIdentifiers::TGenericID test6("mm_device_id", "a1a1a1a1a1a1a1a1a1a1AAAA");                    // correct
        NIdentifiers::TGenericID test7("mm_device_id", "a1a1a1a1a1a1a1a1a1a1");                        // correct

        UNIT_ASSERT_EQUAL(test1.StrictNormalize(), "f1d2aee4fde74e18a6124eab70dc2fcf");
        UNIT_ASSERT_EQUAL(test2.StrictNormalize(), "f1d2aee4fde74e18a6124eab70dc2fcf");
        UNIT_ASSERT_EQUAL(test3.StrictNormalize(), "f1d2aee400004e18a6124eab70dc2fcf");
        UNIT_ASSERT_EQUAL(test4.StrictNormalize(), "00000000111122223333444444444444");
        UNIT_ASSERT_EQUAL(test5.StrictNormalize(), "00000000111122223333444444444444");
        UNIT_ASSERT_EQUAL(test6.StrictNormalize(), "00000000a1a1a1a1a1a1a1a1a1a1aaaa");
        UNIT_ASSERT_EQUAL(test7.StrictNormalize(), "000000000000a1a1a1a1a1a1a1a1a1a1");
    }

    Y_UNIT_TEST(Test_TIdfa_Normalize) {
        NIdentifiers::TIdfa test1("F1D2AEE4-FDE7-4E18-A612-4EAB70DC2FCF");        // correct
        NIdentifiers::TIdfa test2("f1d2aee4-fde7-4e18-a612-4eab70dc2Fcf");        // correct
        NIdentifiers::TIdfa test3("f1d2aee4-0000-4e18-a612-4eab70dc2fcf");        // correct
        NIdentifiers::TIdfa test4("00000000-1111-2222-3333-444444444444");        // correct
        NIdentifiers::TIdfa test5("00000000111122223333444444444444");            // correct

        UNIT_ASSERT_EQUAL(test1.Normalize(), "F1D2AEE4-FDE7-4E18-A612-4EAB70DC2FCF");
        UNIT_ASSERT_EQUAL(test2.Normalize(), "F1D2AEE4-FDE7-4E18-A612-4EAB70DC2FCF");
        UNIT_ASSERT_EQUAL(test3.Normalize(), "F1D2AEE4-0000-4E18-A612-4EAB70DC2FCF");
        UNIT_ASSERT_EQUAL(test4.Normalize(), "00000000-1111-2222-3333-444444444444");
        UNIT_ASSERT_EQUAL(test5.Normalize(), "00000000-1111-2222-3333-444444444444");
    }

    Y_UNIT_TEST(Test_TIdfa_Generic_StrictNormalize) {
        NIdentifiers::TGenericID test1("idfa", "F1D2AEE4-FDE7-4E18-A612-4EAB70DC2FCF");        // correct
        NIdentifiers::TGenericID test2("idfa", "f1d2aee4-fde7-4e18-a612-4eab70dc2Fcf");        // correct
        NIdentifiers::TGenericID test3("idfa", "f1d2aee4-0000-4e18-a612-4eab70dc2fcf");        // correct
        NIdentifiers::TGenericID test4("idfa", "00000000-1111-2222-3333-444444444444");        // correct
        NIdentifiers::TGenericID test5("idfa", "00000000111122223333444444444444");            // correct

        UNIT_ASSERT_EQUAL(test1.Normalize(), "F1D2AEE4-FDE7-4E18-A612-4EAB70DC2FCF");
        UNIT_ASSERT_EQUAL(test2.Normalize(), "F1D2AEE4-FDE7-4E18-A612-4EAB70DC2FCF");
        UNIT_ASSERT_EQUAL(test3.Normalize(), "F1D2AEE4-0000-4E18-A612-4EAB70DC2FCF");
        UNIT_ASSERT_EQUAL(test4.Normalize(), "00000000-1111-2222-3333-444444444444");
        UNIT_ASSERT_EQUAL(test5.Normalize(), "00000000-1111-2222-3333-444444444444");
    }

    Y_UNIT_TEST(Test_TMm_device_id_hash_IsValid) {
        NIdentifiers::TMmDeviceIdHash test1("");                        // not correct, empty
        NIdentifiers::TMmDeviceIdHash test2("42");                      // correct
        NIdentifiers::TMmDeviceIdHash test3("18446744073709551615");    // correct (uint64 max)
        NIdentifiers::TMmDeviceIdHash test4("18446744073709551616");    // not correct (uint64 max +1)
        NIdentifiers::TMmDeviceIdHash test5("018446744073709551610");   // not correct, bad length
        NIdentifiers::TMmDeviceIdHash test6("4294967295");              // correct (uint32 max)


        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::MM_DEVICE_ID_HASH);

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), true);
        UNIT_ASSERT_EQUAL(test3.IsValid(), true);
        UNIT_ASSERT_EQUAL(test4.IsValid(), false);
        UNIT_ASSERT_EQUAL(test5.IsValid(), false);
        UNIT_ASSERT_EQUAL(test6.IsValid(), true);
    }

    Y_UNIT_TEST(Test_TUuid_IsValid) {
        NIdentifiers::TUuid test1("");                                            // not correct, empty
        NIdentifiers::TUuid test2("f1d2aee4-fde7-4e18-a612-4eab70dc2fcf");        // not correct, sep
        NIdentifiers::TUuid test3("f1d2aee4fde74e18a6124eab70dc2fcf");            // correct
        NIdentifiers::TUuid test4("f1d2aee4fde74e18a6124eab70dc2Fcf");            // not correct, upper case
        NIdentifiers::TUuid test5("f1d2aee4-fde7-4e18-a612-4eab70dc2fcfs");       // not correct, sep
        NIdentifiers::TUuid test6("f1d2aee4-fde7-4e18-a612-4ea2fcff1d2-0dc2fcf"); // not correct, sep
        NIdentifiers::TUuid test7("f1d2aee4-fde7-4e18-a612-4ed2-0d2fcf");         // not correct, sep
        NIdentifiers::TUuid test8("f1d2aee4-0000-4e18-a612-4eab70dc2fcf");        // not correct, sep

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::UUID);

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), false);
        UNIT_ASSERT_EQUAL(test3.IsValid(), true);
        UNIT_ASSERT_EQUAL(test4.IsValid(), false);
        UNIT_ASSERT_EQUAL(test5.IsValid(), false);
        UNIT_ASSERT_EQUAL(test6.IsValid(), false);
        UNIT_ASSERT_EQUAL(test7.IsValid(), false);
        UNIT_ASSERT_EQUAL(test8.IsValid(), false);
    }

    Y_UNIT_TEST(Test_TEmail_IsValid) {
        NIdentifiers::TEmail test1("");                       // not correct
        NIdentifiers::TEmail test2("alla_kotlyar@yahoo.com"); // correct
        NIdentifiers::TEmail test3("wdqwdwqd");               // not correct
        NIdentifiers::TEmail test4(".@a.ru");                 // not correct
        NIdentifiers::TEmail test5("test@yandex-team.ru");    // not correct
        NIdentifiers::TEmail test6("yebok88@yandex.ri");      // not correct because is like yandex
        NIdentifiers::TEmail test7("blabla@yandex.ru");       // correct

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::EMAIL);

        NIdentifiers::TEmail test8("test@yandex.ua");       // correct
        NIdentifiers::TEmail test9("test@yandex.uz");       // correct
        NIdentifiers::TEmail test10("test@yandex.by");      // correct
        NIdentifiers::TEmail test11("test@yandex.kz");      // correct
        NIdentifiers::TEmail test12("test@yandex.com");     // correct
        NIdentifiers::TEmail test13("test@yandex.com.tr");  // correct
        NIdentifiers::TEmail test14("test@yandex.com.ge");  // correct
        NIdentifiers::TEmail test15("test@yandex.fr");      // correct
        NIdentifiers::TEmail test16("test@yandex.az");      // correct
        NIdentifiers::TEmail test17("test@yandex.com.am");  // correct
        NIdentifiers::TEmail test18("test@yandex.co.il");   // correct
        NIdentifiers::TEmail test19("test@yandex.kg");      // correct
        NIdentifiers::TEmail test20("test@yandex.lt");      // correct
        NIdentifiers::TEmail test21("test@yandex.lv");      // correct
        NIdentifiers::TEmail test22("test@yandex.md");      // correct
        NIdentifiers::TEmail test23("test@yandex.tj");      // correct
        NIdentifiers::TEmail test24("test@yandex.tm");      // correct
        NIdentifiers::TEmail test25("test@yandex.ee");      // correct
        NIdentifiers::TEmail test26("test@ya.ru");          // correct
        NIdentifiers::TEmail test27("test@m.ya.ru");        // correct
        NIdentifiers::TEmail test28("test@www.ya.ru");      // correct
        NIdentifiers::TEmail test29("test@yandex.ru");      // correct
        NIdentifiers::TEmail test30("te@st@yandex.ru");     // not correct
        NIdentifiers::TEmail test31("test@yandex-team.ru"); // correct
        NIdentifiers::TEmail test32("test@yandex.ru");      // correct
        NIdentifiers::TEmail test33("test@yandex.com.ru");  // correct
        NIdentifiers::TEmail test34("test@yandex.com.ua");  // correct
        NIdentifiers::TEmail test35("test@xyandex.ru");     // not correct because is like yandex
        NIdentifiers::TEmail test36("строка-с-кирилицей@яндекс.рф");     // not correct (no ascii)

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), true);
        UNIT_ASSERT_EQUAL(test3.IsValid(), false);
        UNIT_ASSERT_EQUAL(test4.IsValid(), false);
        UNIT_ASSERT_EQUAL(test5.IsValid(), true);
        UNIT_ASSERT_EQUAL(test6.IsValid(), false);
        UNIT_ASSERT_EQUAL(test7.IsValid(), true);

        UNIT_ASSERT_EQUAL(test8.IsValid(), true);
        UNIT_ASSERT_EQUAL(test9.IsValid(), true);
        UNIT_ASSERT_EQUAL(test10.IsValid(), true);
        UNIT_ASSERT_EQUAL(test11.IsValid(), true);
        UNIT_ASSERT_EQUAL(test12.IsValid(), true);
        UNIT_ASSERT_EQUAL(test13.IsValid(), true);
        UNIT_ASSERT_EQUAL(test14.IsValid(), true);
        UNIT_ASSERT_EQUAL(test15.IsValid(), true);
        UNIT_ASSERT_EQUAL(test16.IsValid(), true);
        UNIT_ASSERT_EQUAL(test17.IsValid(), true);
        UNIT_ASSERT_EQUAL(test18.IsValid(), true);
        UNIT_ASSERT_EQUAL(test19.IsValid(), true);
        UNIT_ASSERT_EQUAL(test20.IsValid(), true);
        UNIT_ASSERT_EQUAL(test21.IsValid(), true);
        UNIT_ASSERT_EQUAL(test22.IsValid(), true);
        UNIT_ASSERT_EQUAL(test23.IsValid(), true);
        UNIT_ASSERT_EQUAL(test24.IsValid(), true);
        UNIT_ASSERT_EQUAL(test25.IsValid(), true);
        UNIT_ASSERT_EQUAL(test26.IsValid(), true);
        UNIT_ASSERT_EQUAL(test27.IsValid(), true);
        UNIT_ASSERT_EQUAL(test28.IsValid(), true);
        UNIT_ASSERT_EQUAL(test29.IsValid(), true);
        UNIT_ASSERT_EQUAL(test30.IsValid(), false);
        UNIT_ASSERT_EQUAL(test31.IsValid(), true);
        UNIT_ASSERT_EQUAL(test32.IsValid(), true);
        UNIT_ASSERT_EQUAL(test33.IsValid(), true);
        UNIT_ASSERT_EQUAL(test34.IsValid(), true);
        UNIT_ASSERT_EQUAL(test35.IsValid(), false);
        UNIT_ASSERT_EQUAL(test36.IsValid(), false);
    }

    Y_UNIT_TEST(Test_TEmail_Normalize) {
        NIdentifiers::TEmail test1("andrey@yandex.ru");    // not correct, empty
        NIdentifiers::TEmail test2("andrey@yandex.com");   // correct
        NIdentifiers::TEmail test3("andrey@01yandex.kz");  // not correct
        NIdentifiers::TEmail test4(".@a.ru");              // not correct
        NIdentifiers::TEmail test5("test@yandex-team.ru"); // correct
        NIdentifiers::TEmail test6("yebok88@yandex.ri");   // correct
        NIdentifiers::TEmail test7("test@googlemail.com"); // correct

        NIdentifiers::TEmail test8("test@yandex.ua");       // correct
        NIdentifiers::TEmail test9("test@yandex.uz");       // correct
        NIdentifiers::TEmail test10("test@yandex.by");      // correct
        NIdentifiers::TEmail test11("test@yandex.kz");      // correct
        NIdentifiers::TEmail test12("test@yandex.com");     // correct
        NIdentifiers::TEmail test13("test@yandex.com.tr");  // correct
        NIdentifiers::TEmail test14("test@yandex.com.ge");  // correct
        NIdentifiers::TEmail test15("test@yandex.fr");      // correct
        NIdentifiers::TEmail test16("test@yandex.az");      // correct
        NIdentifiers::TEmail test17("test@yandex.com.am");  // correct
        NIdentifiers::TEmail test18("test@yandex.co.il");   // correct
        NIdentifiers::TEmail test19("test@yandex.kg");      // correct
        NIdentifiers::TEmail test20("test@yandex.lt");      // correct
        NIdentifiers::TEmail test21("test@yandex.lv");      // correct
        NIdentifiers::TEmail test22("test@yandex.md");      // correct
        NIdentifiers::TEmail test23("test@yandex.tj");      // correct
        NIdentifiers::TEmail test24("test@yandex.tm");      // correct
        NIdentifiers::TEmail test25("test@yandex.ee");      // correct
        NIdentifiers::TEmail test26("test@ya.ru");          // correct
        NIdentifiers::TEmail test27("test@yandex-team.ru"); // correct
        NIdentifiers::TEmail test28("test@yandex.com.ru");  // correct
        NIdentifiers::TEmail test29("test@yandex.com.ua");  // correct
        NIdentifiers::TEmail test30("строка-с-кирилицей@яндекс.рф");     // not correct (no ascii)
        NIdentifiers::TEmail test31("test+suffix@email.com");  // correct (+suffix)
        NIdentifiers::TEmail test32("+71231234567@e.com");  // correct (+phone)
        NIdentifiers::TEmail test33("++++++++++++++++evgeni-pinchuk@tut.by");  // correct (++)

        NIdentifiers::TEmail test71("test.with.dots.g@googlemail.com"); // correct
        NIdentifiers::TEmail test72("test.with.dots.g@gMaIl.com"); // correct
        NIdentifiers::TEmail test73("test.with.dots.g@YaNdEx.com"); // correct

        NIdentifiers::TEmail test74("test.with.......multi.dots.g@googlemail.com"); // correct
        NIdentifiers::TEmail test75("test.with.......multi.dots.g@gMaIl.com"); // correct
        NIdentifiers::TEmail test76("test.with.......multi.dots.g@YaNdEx.com"); // correct

        UNIT_ASSERT_EQUAL(test1.Normalize(), "andrey@yandex.ru");
        UNIT_ASSERT_EQUAL(test2.Normalize(), "andrey@yandex.ru");
        UNIT_ASSERT_EQUAL(test3.Normalize(), "andrey@01yandex.kz");
        UNIT_ASSERT_EQUAL(test4.Normalize(), ".@a.ru");
        UNIT_ASSERT_EQUAL(test5.Normalize(), "test@yandex-team.ru");
        UNIT_ASSERT_EQUAL(test6.Normalize(), "yebok88@yandex.ri");
        UNIT_ASSERT_EQUAL(test7.Normalize(), "test@gmail.com");

        UNIT_ASSERT_EQUAL(test8.Normalize(), "test@yandex.ru");
        UNIT_ASSERT_EQUAL(test9.Normalize(), "test@yandex.ru");
        UNIT_ASSERT_EQUAL(test10.Normalize(), "test@yandex.ru");
        UNIT_ASSERT_EQUAL(test11.Normalize(), "test@yandex.ru");
        UNIT_ASSERT_EQUAL(test12.Normalize(), "test@yandex.ru");
        UNIT_ASSERT_EQUAL(test13.Normalize(), "test@yandex.ru");
        UNIT_ASSERT_EQUAL(test14.Normalize(), "test@yandex.ru");
        UNIT_ASSERT_EQUAL(test15.Normalize(), "test@yandex.ru");
        UNIT_ASSERT_EQUAL(test16.Normalize(), "test@yandex.ru");
        UNIT_ASSERT_EQUAL(test17.Normalize(), "test@yandex.ru");
        UNIT_ASSERT_EQUAL(test18.Normalize(), "test@yandex.ru");
        UNIT_ASSERT_EQUAL(test19.Normalize(), "test@yandex.ru");
        UNIT_ASSERT_EQUAL(test20.Normalize(), "test@yandex.ru");
        UNIT_ASSERT_EQUAL(test21.Normalize(), "test@yandex.ru");
        UNIT_ASSERT_EQUAL(test22.Normalize(), "test@yandex.ru");
        UNIT_ASSERT_EQUAL(test23.Normalize(), "test@yandex.ru");
        UNIT_ASSERT_EQUAL(test24.Normalize(), "test@yandex.ru");
        UNIT_ASSERT_EQUAL(test25.Normalize(), "test@yandex.ru");
        UNIT_ASSERT_EQUAL(test26.Normalize(), "test@yandex.ru");
        UNIT_ASSERT_EQUAL(test27.Normalize(), "test@yandex-team.ru");
        UNIT_ASSERT_EQUAL(test28.Normalize(), "test@yandex.ru");
        UNIT_ASSERT_EQUAL(test29.Normalize(), "test@yandex.ru");
        UNIT_ASSERT_EQUAL(test30.Normalize(), "строка-с-кирилицей@яндекс.рф");
        UNIT_ASSERT_EQUAL(test31.Normalize(), "test@email.com");
        UNIT_ASSERT_EQUAL(test32.Normalize(), "71231234567@e.com");
        UNIT_ASSERT_EQUAL(test33.Normalize(), "evgeni-pinchuk@tut.by");

        UNIT_ASSERT_EQUAL(test71.Normalize(), "testwithdotsg@gmail.com");
        UNIT_ASSERT_EQUAL(test72.Normalize(), "testwithdotsg@gmail.com");
        UNIT_ASSERT_EQUAL(test73.Normalize(), "test-with-dots-g@yandex.ru");
        UNIT_ASSERT_EQUAL(test74.Normalize(), "testwithmultidotsg@gmail.com");
        UNIT_ASSERT_EQUAL(test75.Normalize(), "testwithmultidotsg@gmail.com");
        UNIT_ASSERT_EQUAL(test76.Normalize(), "test-with-multi-dots-g@yandex.ru");
    }

    Y_UNIT_TEST(Test_TEmail_IsUnnormalEmail) {
        TEmailTest test1("andrey@yandex.ru");
        TEmailTest test2("andrey@yandex.kz");
        TEmailTest test3("andrey@yandex.ua");
        TEmailTest test4("andrey@yandex.ruu");
        TEmailTest test5("andrey@01yandex.ruua");
        TEmailTest test6("andrey@01yandex.comtr");

        TEmailTest test7("andrey@ya.ru");
        TEmailTest test8("andrey@ya.kz");
        TEmailTest test9("andrey@ya.ua");
        TEmailTest test10("andrey@ya.ruu");
        TEmailTest test11("andrey@01ya.ruua");

        UNIT_ASSERT_EQUAL(test1.WrapIsUnnormalEmail(), false);
        UNIT_ASSERT_EQUAL(test2.WrapIsUnnormalEmail(), true);
        UNIT_ASSERT_EQUAL(test3.WrapIsUnnormalEmail(), true);
        UNIT_ASSERT_EQUAL(test4.WrapIsUnnormalEmail(), true);
        UNIT_ASSERT_EQUAL(test5.WrapIsUnnormalEmail(), true);
        UNIT_ASSERT_EQUAL(test6.WrapIsUnnormalEmail(), true);

        UNIT_ASSERT_EQUAL(test7.WrapIsUnnormalEmail(), false);
        UNIT_ASSERT_EQUAL(test8.WrapIsUnnormalEmail(), true);
        UNIT_ASSERT_EQUAL(test9.WrapIsUnnormalEmail(), true);
        UNIT_ASSERT_EQUAL(test10.WrapIsUnnormalEmail(), true);
        UNIT_ASSERT_EQUAL(test11.WrapIsUnnormalEmail(), true);
    }

    Y_UNIT_TEST(Test_TEmail_IsValidHeuristic) {
        TEmailTest test1("andrey@yandex.ru");
        TEmailTest test2("andrey@yandex.kz");
        TEmailTest test3("andrey@yandex.ua");
        TEmailTest test4("andrey@yandex.ruu");
        TEmailTest test5("andrey@01yandex.ruua");
        TEmailTest test6("andrey@yandex.com.tr");

        UNIT_ASSERT_EQUAL(test1.WrapIsValidHeuristic(), true);
        UNIT_ASSERT_EQUAL(test2.WrapIsValidHeuristic(), true);
        UNIT_ASSERT_EQUAL(test3.WrapIsValidHeuristic(), true);
        UNIT_ASSERT_EQUAL(test4.WrapIsValidHeuristic(), false);
        UNIT_ASSERT_EQUAL(test5.WrapIsValidHeuristic(), false);
        UNIT_ASSERT_EQUAL(test6.WrapIsValidHeuristic(), true);
    }

    Y_UNIT_TEST(Test_TEmail_GetMd5) {
        NIdentifiers::TEmail test1("hasanurhan65@yandex.ru");
        NIdentifiers::TEmail test2("hasanurhan@yandex.ru");
        NIdentifiers::TEmail test3("hasanuri@gmail.com");

        UNIT_ASSERT_STRINGS_EQUAL(test1.GetMd5(), "2194c06f085b5f489d9956b4517f7f39");
        UNIT_ASSERT_STRINGS_EQUAL(test2.GetMd5(), "340b88a9f589e9301af2b3b256513086");
        UNIT_ASSERT_STRINGS_EQUAL(test3.GetMd5(), "2722ba0a82ba5bf6c6b1af66096bd16d");
    }

    Y_UNIT_TEST(Test_TMacExt_GetMd5) {
        NIdentifiers::TMacExt test("11:22:33:44:55:66");

        UNIT_ASSERT_STRINGS_EQUAL(test.GetMd5(), "86E65B1EF4A830AF347AC05AB4F0E999");
    }

    Y_UNIT_TEST(Test_TEmail_GetSha256) {
        NIdentifiers::TEmail test1("hasanurhan65@yandex.ru");
        NIdentifiers::TEmail test2("hasanurhan@yandex.ru");
        NIdentifiers::TEmail test3("hasanuri@gmail.com");

        UNIT_ASSERT_STRINGS_EQUAL(test1.GetSha256(),
                                  "142ad76a39223eec4d8a3ad6cc8641560801ceb22ef20cd7d9eda03d83316a70");
        UNIT_ASSERT_STRINGS_EQUAL(test2.GetSha256(),
                                  "5479e8f6b12d8e00c50893e2d2e0af1bc112555569adeabb2fe946d473967f28");
        UNIT_ASSERT_STRINGS_EQUAL(test3.GetSha256(),
                                  "104f2127f4297f34fb2d7a4b7a13cf4096d7c8b81c6c89f6935768c9faf0e8d1");
    }

    Y_UNIT_TEST(Test_TIcoookie_IsValid) {
        NIdentifiers::TIcookie test1("");                    // not correct, empty
        NIdentifiers::TIcookie test2("0");                   // nor correct, less than min
        NIdentifiers::TIcookie test3("-1");                  // not correct, negative number
        NIdentifiers::TIcookie test4("10113701529442803");   // correct
        NIdentifiers::TIcookie test5("0011188541530035229"); // not correct, start with 0
        NIdentifiers::TIcookie test6("10113701563508333");   // correct, from future
        NIdentifiers::TIcookie test7("8322285710565070336");   // correct, from uuid


        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::ICOOKIE);

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), false);
        UNIT_ASSERT_EQUAL(test3.IsValid(), false);
        UNIT_ASSERT_EQUAL(test4.IsValid(), true);
        UNIT_ASSERT_EQUAL(test5.IsValid(), false);
        UNIT_ASSERT_EQUAL(test5.Value(), "0011188541530035229");
        UNIT_ASSERT_EQUAL(test6.IsValid(), true);
        UNIT_ASSERT_EQUAL(test7.IsValid(), true);
    }

    Y_UNIT_TEST(Test_TPhone_IsValid) {
        const auto TestPhone = [](const TString& raw_phone, const bool is_valid, const TString& norm) {
            NIdentifiers::TPhone phone(raw_phone);
            UNIT_ASSERT_EQUAL(phone.GetType(), EIdType::PHONE);
            UNIT_ASSERT_EQUAL(phone.IsValid(), is_valid);
            UNIT_ASSERT_EQUAL(phone.Normalize(), norm);
        };

        TestPhone("", false, "");     // not correct, empty
        TestPhone("0", false, "0");   // not correct, less than min
        TestPhone("-1", false, "-1"); // not correct, negative number
        TestPhone("+89168751594", false, "+89168751594");       // not correct, start with +8
        TestPhone("+79168751593", true, "+79168751593");        // correct
        TestPhone("+7 916 875 15 93", true, "+79168751593");    // correct
        TestPhone("+7(916)-875-15-93", true, "+79168751593");   // correct
        TestPhone("+7(916) 875 15 93", true, "+79168751593");   // correct
        TestPhone("8 (916) 875 15 93", true, "+79168751593");   // correct
        TestPhone("79165332388", true, "+79165332388");         // correct
        TestPhone("+998931868848", true, "+998931868848");      // allow non russian numbers
        TestPhone("+1 345 657 1234", true, "+13456571234");     // allow non russian numbers
    }

    Y_UNIT_TEST(Test_TPhone_Value_Normalize) {
        const auto TestPhone = [](const TString& raw_phone, const TString& normalized) {
            NIdentifiers::TPhone phone(raw_phone);
            UNIT_ASSERT(phone.IsValid());
            UNIT_ASSERT_STRINGS_EQUAL(phone.Value(), raw_phone);
            UNIT_ASSERT_STRINGS_EQUAL(phone.Normalize(), normalized);
        };

        TestPhone("+79168751593", "+79168751593");
        TestPhone("+7 916 875 15 93", "+79168751593");
        TestPhone("+7(916)-875-15-93", "+79168751593");
        TestPhone("+7(916) 875 15 93", "+79168751593");
        TestPhone("8(916) 875 15 93", "+79168751593");
        TestPhone("89168751593", "+79168751593");
        TestPhone("11111111111111111", "+11111111111");
    }

    Y_UNIT_TEST(Test_TPhone_GetMd5) {
        const auto TestPhone = [](const TString& raw_phone, const TString& valid_hash) {
            NIdentifiers::TPhone phone(raw_phone);
            UNIT_ASSERT_STRINGS_EQUAL(phone.GetMd5(), valid_hash);
        };

        TestPhone("+79183938034", "3ac5b42a336f051a3295eb6d6266c32b");
        TestPhone("+79183938035", "99e421d0f4183866e987eaacecee76b1");
        TestPhone("+79183938036", "ed36c9dc68c8bc1a98b31937d27e0e59");
    }

    Y_UNIT_TEST(Test_TPhone_GetSha256) {
        const auto TestPhone = [](const TString& raw_phone, const TString& valid_hash) {
            NIdentifiers::TPhone phone(raw_phone);
            UNIT_ASSERT_STRINGS_EQUAL(phone.GetSha256(), valid_hash);
        };

        TestPhone("+79183938034",
                  "5085bb1313fafdd2da5f16b22cb0e57d22e855223dc1d16f2de04bd7e4440c80");
        TestPhone("+79183938035",
                  "7f1cf9f0367b331623448be448976e408ce53a70a1b7fff8edeb4d724e315a66");
        TestPhone("+79183938036",
                  "1ac87020bab6dd6e82ddfb9227e4523ec009370d92f311af0d906b466adbdef7");
    }

    Y_UNIT_TEST(TEST_md5) {
        NIdentifiers::TMd5 test1("0123456789abcdefABCDEF9876543210");  // correct
        NIdentifiers::TMd5 test2("0123456789abcdefABCDEF98765432100"); // not correct
        NIdentifiers::TMd5 test3("0123456789qbcdefABCDEF9876543210");  // not correct

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::MD5);

        UNIT_ASSERT_EQUAL(test1.IsValid(), true);
        UNIT_ASSERT_EQUAL(test2.IsValid(), false);
        UNIT_ASSERT_EQUAL(test3.IsValid(), false);
    }

    Y_UNIT_TEST(TEST_email_md5) {
        NIdentifiers::TEmailMd5 test1("0123456789abcdefABCDEF9876543210");  // correct
        NIdentifiers::TEmailMd5 test2("0123456789abcdefABCDEF98765432100"); // not correct
        NIdentifiers::TEmailMd5 test3("0123456789qbcdefABCDEF9876543210");  // not correct

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::EMAIL_MD5);

        UNIT_ASSERT_EQUAL(test1.IsValid(), true);
        UNIT_ASSERT_EQUAL(test2.IsValid(), false);
        UNIT_ASSERT_EQUAL(test3.IsValid(), false);
    }

    Y_UNIT_TEST(TEST_phone_md5) {
        NIdentifiers::TPhoneMd5 test1("0123456789abcdefABCDEF9876543210");  // correct
        NIdentifiers::TPhoneMd5 test2("0123456789abcdefABCDEF98765432100"); // not correct
        NIdentifiers::TPhoneMd5 test3("0123456789qbcdefABCDEF9876543210");  // not correct

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::PHONE_MD5);

        UNIT_ASSERT_EQUAL(test1.IsValid(), true);
        UNIT_ASSERT_EQUAL(test2.IsValid(), false);
        UNIT_ASSERT_EQUAL(test3.IsValid(), false);
    }

    Y_UNIT_TEST(TEST_sha256) {
        NIdentifiers::TSha256 test1("0123456789abcdefABCDEF98765432100123456789abcdefABCDEF9876543210");   // correct
        NIdentifiers::TSha256 test2("0123456789abcdefABCDEF987654321000123456789abcdefABCDEF98765432100"); // not correct
        NIdentifiers::TSha256 test3("0123456789qbcdefABCDEF98765432100123456789qbcdefABCDEF9876543210");   // not correct

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::SHA256);

        UNIT_ASSERT_EQUAL(test1.IsValid(), true);
        UNIT_ASSERT_EQUAL(test2.IsValid(), false);
        UNIT_ASSERT_EQUAL(test3.IsValid(), false);
    }

    Y_UNIT_TEST(TEST_email_sha256) {
        NIdentifiers::TEmailSha256 test1("0123456789abcdefABCDEF98765432100123456789abcdefABCDEF9876543210");   // correct
        NIdentifiers::TEmailSha256 test2("0123456789abcdefABCDEF987654321000123456789abcdefABCDEF98765432100"); // not correct
        NIdentifiers::TEmailSha256 test3("0123456789qbcdefABCDEF98765432100123456789qbcdefABCDEF9876543210");   // not correct

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::EMAIL_SHA256);

        UNIT_ASSERT_EQUAL(test1.IsValid(), true);
        UNIT_ASSERT_EQUAL(test2.IsValid(), false);
        UNIT_ASSERT_EQUAL(test3.IsValid(), false);
    }

    Y_UNIT_TEST(TEST_phone_sha256) {
        NIdentifiers::TPhoneSha256 test1("0123456789abcdefABCDEF98765432100123456789abcdefABCDEF9876543210");   // correct
        NIdentifiers::TPhoneSha256 test2("0123456789abcdefABCDEF987654321000123456789abcdefABCDEF98765432100"); // not correct
        NIdentifiers::TPhoneSha256 test3("0123456789qbcdefABCDEF98765432100123456789qbcdefABCDEF9876543210");   // not correct

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::PHONE_SHA256);

        UNIT_ASSERT_EQUAL(test1.IsValid(), true);
        UNIT_ASSERT_EQUAL(test2.IsValid(), false);
        UNIT_ASSERT_EQUAL(test3.IsValid(), false);
    }

    Y_UNIT_TEST(TEST_mac_isvalid) {
        NIdentifiers::TMac test1("00:11:22:33:ee:c1"); // correct
        NIdentifiers::TMac test2("11:FF:22:AA:33:CC"); // correct
        NIdentifiers::TMac test3("001122334455");      // correct
        NIdentifiers::TMac test4("00:11:22:33:ee:u1"); // not correct
        NIdentifiers::TMac test5("0011222334455");     // not correct
        NIdentifiers::TMac test6("00:00:00:00:00:01"); // correct
        NIdentifiers::TMac test7("00-00-00-00-00-01"); // correct
        NIdentifiers::TMac test8("00:00:00:00:0021");  // correct

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::MAC);

        UNIT_ASSERT_EQUAL(test1.IsValid(), true);
        UNIT_ASSERT_EQUAL(test2.IsValid(), true);
        UNIT_ASSERT_EQUAL(test3.IsValid(), true);
        UNIT_ASSERT_EQUAL(test4.IsValid(), false);
        UNIT_ASSERT_EQUAL(test5.IsValid(), false);
        UNIT_ASSERT_EQUAL(test6.IsValid(), true);
        UNIT_ASSERT_EQUAL(test7.IsValid(), true);
        UNIT_ASSERT_EQUAL(test8.IsValid(), false);
    }

    Y_UNIT_TEST(TEST_mac_normalize) {
        NIdentifiers::TMac test1("00:11:22:33:ee:c1"); // correct
        NIdentifiers::TMac test2("11:FF:22:AA:33:CC"); // correct
        NIdentifiers::TMac test3("001122334455");      // correct
        NIdentifiers::TMac test4("00:00:00:00:00:01"); // correct
        NIdentifiers::TMac test5("00-00-00-00-00-01"); // correct

        UNIT_ASSERT_EQUAL(test1.Normalize(), "00:11:22:33:ee:c1");
        UNIT_ASSERT_EQUAL(test2.Normalize(), "11:ff:22:aa:33:cc");
        UNIT_ASSERT_EQUAL(test3.Normalize(), "00:11:22:33:44:55");
        UNIT_ASSERT_EQUAL(test4.Normalize(), "00:00:00:00:00:01");
        UNIT_ASSERT_EQUAL(test5.Normalize(), "00:00:00:00:00:01");
    }

    Y_UNIT_TEST(TEST_login_IsValid) {
        NIdentifiers::TLogin test1("");                                      // not correct
        NIdentifiers::TLogin test2("login");                                 // correct
        NIdentifiers::TLogin test3("  login@ya.ru ");                        // correct
        NIdentifiers::TLogin test4("login@ya@gmail.com");                    // not correct
        NIdentifiers::TLogin test5("4login");                                // not correct
        NIdentifiers::TLogin test6(".login");                                // not correct
        NIdentifiers::TLogin test7("login.");                                // not correct
        NIdentifiers::TLogin test8("-login");                                // not correct
        NIdentifiers::TLogin test9("login-");                                // not correct
        NIdentifiers::TLogin test10("log-in");                               // correct
        NIdentifiers::TLogin test11("log.-in");                              // not correct
        NIdentifiers::TLogin test12("log-.in");                              // not correct
        NIdentifiers::TLogin test13("log..in");                              // not correct
        NIdentifiers::TLogin test14("log--in");                              // not correct
        NIdentifiers::TLogin test15("login@gmail.com");                      // correct
        NIdentifiers::TLogin test16("6c48667f-99c2-4bd8-8bf6-aa8a02763e2c"); // not correct
        NIdentifiers::TLogin test17("ac48667f-99c2-4bd8-8bf6-aa8a02763e2c"); // not correct
        NIdentifiers::TLogin test18("парк@voliacable.com");                  // not correct

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::LOGIN);

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), true);
        UNIT_ASSERT_EQUAL(test3.IsValid(), true);
        UNIT_ASSERT_EQUAL(test4.IsValid(), false);
        UNIT_ASSERT_EQUAL(test5.IsValid(), false);
        UNIT_ASSERT_EQUAL(test6.IsValid(), false);
        UNIT_ASSERT_EQUAL(test7.IsValid(), false);
        UNIT_ASSERT_EQUAL(test8.IsValid(), false);
        UNIT_ASSERT_EQUAL(test9.IsValid(), false);
        UNIT_ASSERT_EQUAL(test10.IsValid(), true);
        UNIT_ASSERT_EQUAL(test11.IsValid(), false);
        UNIT_ASSERT_EQUAL(test12.IsValid(), false);
        UNIT_ASSERT_EQUAL(test13.IsValid(), false);
        UNIT_ASSERT_EQUAL(test14.IsValid(), false);
        UNIT_ASSERT_EQUAL(test15.IsValid(), true);
        UNIT_ASSERT_EQUAL(test16.IsValid(), false);
        UNIT_ASSERT_EQUAL(test17.IsValid(), false);
        UNIT_ASSERT_EQUAL(test18.IsValid(), false);
    }

    Y_UNIT_TEST(TEST_login_Normalize) {
        NIdentifiers::TLogin test1("");                   // not correct
        NIdentifiers::TLogin test2("  loGin  ");          // correct
        NIdentifiers::TLogin test3(" log.in@ya.ru  ");    // correct
        NIdentifiers::TLogin test4("login@ya@gmail.com"); // not correct
        NIdentifiers::TLogin test5("4login");             // not correct
        NIdentifiers::TLogin test6(".login");             // not correct
        NIdentifiers::TLogin test7("login.");             // not correct
        NIdentifiers::TLogin test8("-login");             // not correct
        NIdentifiers::TLogin test9("login-");             // not correct
        NIdentifiers::TLogin test10("log.in");            // correct
        NIdentifiers::TLogin test11("log.-in");           // not correct
        NIdentifiers::TLogin test12("log-.in");           // not correct
        NIdentifiers::TLogin test13("log..in");           // not correct
        NIdentifiers::TLogin test14("log--in");           // not correct
        NIdentifiers::TLogin test15("login@gmail.com");   // not correct

        UNIT_ASSERT_EQUAL(test1.Normalize(), "");
        UNIT_ASSERT_EQUAL(test2.Normalize(), "login");
        UNIT_ASSERT_EQUAL(test3.Normalize(), "log-in");
        UNIT_ASSERT_EQUAL(test4.Normalize(), "login@ya@gmail.com");
        UNIT_ASSERT_EQUAL(test5.Normalize(), "4login");
        UNIT_ASSERT_EQUAL(test6.Normalize(), ".login");
        UNIT_ASSERT_EQUAL(test7.Normalize(), "login.");
        UNIT_ASSERT_EQUAL(test8.Normalize(), "-login");
        UNIT_ASSERT_EQUAL(test9.Normalize(), "login-");
        UNIT_ASSERT_EQUAL(test10.Normalize(), "log-in");
        UNIT_ASSERT_EQUAL(test11.Normalize(), "log.-in");
        UNIT_ASSERT_EQUAL(test12.Normalize(), "log-.in");
        UNIT_ASSERT_EQUAL(test13.Normalize(), "log..in");
        UNIT_ASSERT_EQUAL(test14.Normalize(), "log--in");
        UNIT_ASSERT_EQUAL(test15.Normalize(), "login@gmail.com");
    }

    Y_UNIT_TEST(TEST_vk_id) {
        NIdentifiers::TVkId test1("");
        NIdentifiers::TVkId test2("0");
        NIdentifiers::TVkId test3("1234");
        NIdentifiers::TVkId test4("001234");
        NIdentifiers::TVkId test5("73645sdas");

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::VK_ID);

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), false);
        UNIT_ASSERT_EQUAL(test3.IsValid(), true);
        UNIT_ASSERT_EQUAL(test4.IsValid(), true);
        UNIT_ASSERT_EQUAL(test5.IsValid(), true);
    }

    Y_UNIT_TEST(TEST_vk_name) {
        NIdentifiers::TVkName test1("");
        NIdentifiers::TVkName test2("abacaba");

        UNIT_ASSERT_EQUAL(test1.GetType(), NCrypta::NIdentifiersProto::NIdType::EIdType::VK_NAME);

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), true);
    }

    Y_UNIT_TEST(TEST_ok_id) {
        NIdentifiers::TOkId test1("");
        NIdentifiers::TOkId test2("0");
        NIdentifiers::TOkId test3("1234");
        NIdentifiers::TOkId test4("001234");
        NIdentifiers::TOkId test5("73645sdas");

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::OK_ID);

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), false);
        UNIT_ASSERT_EQUAL(test3.IsValid(), true);
        UNIT_ASSERT_EQUAL(test4.IsValid(), true);
        UNIT_ASSERT_EQUAL(test5.IsValid(), false);
    }

    Y_UNIT_TEST(Test_TDuid_IsValid) {
        NIdentifiers::TDuid test1("");                     // not correct, empty
        NIdentifiers::TDuid test2("0");                    // not correct, size < 11
        NIdentifiers::TDuid test3("12341234");             // not correct, size < 11
        NIdentifiers::TDuid test4("-1570655981751179923");   // not correct, negative number
        NIdentifiers::TDuid test5("15706559817511792a3");    // not correct, not the number
        NIdentifiers::TDuid test6("1570655981751179923");    // correct
        NIdentifiers::TDuid test7("1970655981751179923");    // not correct - ts in the future

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::DUID);

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), false);
        UNIT_ASSERT_EQUAL(test3.IsValid(), false);
        UNIT_ASSERT_EQUAL(test4.IsValid(), false);
        UNIT_ASSERT_EQUAL(test5.IsValid(), false);
        UNIT_ASSERT_EQUAL(test6.IsValid(), true);
        UNIT_ASSERT_EQUAL(test7.IsValid(), false);
    }

    Y_UNIT_TEST(Test_unversal_proto) {
        // Should check is generic id correct construct from proto
        {
            // 64 bit case
            TestGenericIdProto<NIdentifiers::TYandexuid>();
            TestGenericIdProto<NIdentifiers::TIcookie>();
            TestGenericIdProto<NIdentifiers::TKpId>();
            TestGenericIdProto<NIdentifiers::TMac>();
            TestGenericIdProto<NIdentifiers::TPuid>();
            TestGenericIdProto<NIdentifiers::TDirectClientId>();
        }
        {
            // 128 bit case
            TestGenericIdProto<NIdentifiers::TIdfa>();
            TestGenericIdProto<NIdentifiers::TGaid>();
            TestGenericIdProto<NIdentifiers::TOaid>();
            TestGenericIdProto<NIdentifiers::TMd5>();
        }
        {
            // 256 bit case
            TestGenericIdProto<NIdentifiers::TSha256>();
        }
        {
            // string case
            TestGenericIdProto<NIdentifiers::TLogin>();
            TestGenericIdProto<NIdentifiers::TEmail>();
            TestGenericIdProto<NIdentifiers::TPartnerRecordId>();
            TestGenericIdProto<NIdentifiers::TVkName>();
        }
    }

    Y_UNIT_TEST(Test_unversal) {
        NIdentifiers::TGenericID test1("Yandexuid"); // not correct, empty
        NIdentifiers::TGenericID test2("Yandexuid"); // not correct, less than min
        NIdentifiers::TGenericID test3("Yandexuid"); // not correct, negative number
        NIdentifiers::TGenericID test4("Yandexuid"); // correct
        NIdentifiers::TGenericID test5("Yandexuid"); // not correct, start with 0

        test1.SetValue("");
        test2.SetValue("0");
        test3.SetValue("-1");
        test4.SetValue("10113701529442803");
        test5.SetValue("0011188541530035229");

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), false);
        UNIT_ASSERT_EQUAL(test3.IsValid(), false);
        UNIT_ASSERT_EQUAL(test4.IsValid(), true);
        UNIT_ASSERT_EQUAL(test5.IsValid(), true);

        UNIT_ASSERT_EQUAL(test5.Normalize(), "11188541530035229");
        UNIT_ASSERT_EQUAL(test5.GetTypeString(), "yandexuid");
    }

    Y_UNIT_TEST(TEST_universal2) {
        NIdentifiers::TGenericID test1(NCrypta::NIdentifiersProto::NIdType::OK_ID, "");
        NIdentifiers::TGenericID test2(NCrypta::NIdentifiersProto::NIdType::OK_ID, "1234");
        NIdentifiers::TGenericID test3(NCrypta::NIdentifiersProto::NIdType::OK_ID, "001234");
        NIdentifiers::TGenericID test4(NCrypta::NIdentifiersProto::NIdType::OK_ID, "73645sdas");
        NIdentifiers::TGenericID test5(NCrypta::NIdentifiersProto::NIdType::DIRECT_CLIENT_ID, "15456464");
        NIdentifiers::TGenericID test6("email", "test@test.com");
        NIdentifiers::TGenericID test7("phone", "+79168751595");

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), true);
        UNIT_ASSERT_EQUAL(test3.IsValid(), true);
        UNIT_ASSERT_EQUAL(test4.IsValid(), false);
        UNIT_ASSERT_EQUAL(test5.IsValid(), true);
        UNIT_ASSERT_EQUAL(test6.GetMd5(), "b642b4217b34b1e8d3bd915fc65c4452");
        UNIT_ASSERT_EQUAL(test6.GetSha256(),
                          "f660ab912ec121d1b1e928a0bb4bc61b15f5ad44d5efdc4e1c92a25e99b8e44a");
        UNIT_ASSERT_EQUAL(test7.GetMd5(), "801ad0017c8b12f3f53c90cd6aa1350c");
        UNIT_ASSERT_EQUAL(test7.GetSha256(),
                          "647ff9f573b2637b08fcc16f36b0b0ae3df560c820fa2a99647024e28c8298a8");

        UNIT_ASSERT_EQUAL(test1.GetTypeString(), "ok_id");
        UNIT_ASSERT_EQUAL(test6.GetTypeString(), "email");
        UNIT_ASSERT_EQUAL(test7.GetTypeString(), "phone");
    }

    Y_UNIT_TEST(TEST_macExt_universal) {
        NIdentifiers::TGenericID macExt("mac_ext", "11:22:33:44:55:66");
        UNIT_ASSERT_EQUAL(macExt.GetType(), NCrypta::NIdentifiersProto::NIdType::MAC_EXT);
        UNIT_ASSERT(macExt.IsValid());

        NIdentifiers::TGenericID macExtMd5("mac_ext_md5", macExt.GetMd5());
        UNIT_ASSERT_EQUAL(macExtMd5.GetType(), NCrypta::NIdentifiersProto::NIdType::MAC_EXT_MD5);
        UNIT_ASSERT(macExtMd5.IsValid());
    }

    Y_UNIT_TEST(TEST_cryptaid) {
        NIdentifiers::TCryptaId test1("");
        NIdentifiers::TCryptaId test2("1234");
        NIdentifiers::TCryptaId test3("001234");
        NIdentifiers::TCryptaId test4("73645sdas");

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::CRYPTA_ID);

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), true);
        UNIT_ASSERT_EQUAL(test3.IsValid(), true);
        UNIT_ASSERT_EQUAL(test4.IsValid(), false);
    }

    Y_UNIT_TEST(TEST_fb_id) {
        NIdentifiers::TFbId test1("");
        NIdentifiers::TFbId test2("1234");
        NIdentifiers::TFbId test3("001234");
        NIdentifiers::TFbId test4("73645sdas");

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::FB_ID);

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), true);
        UNIT_ASSERT_EQUAL(test3.IsValid(), true);
        UNIT_ASSERT_EQUAL(test4.IsValid(), false);
    }

    Y_UNIT_TEST(TEST_avito_id) {
        NIdentifiers::TAvitoId test1("");
        NIdentifiers::TAvitoId test2("1234");
        NIdentifiers::TAvitoId test3("001234");
        NIdentifiers::TAvitoId test4("73645sdas");

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::AVITO_ID);

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), true);
        UNIT_ASSERT_EQUAL(test3.IsValid(), true);
        UNIT_ASSERT_EQUAL(test4.IsValid(), false);
    }

    Y_UNIT_TEST(TEST_kp_id) {
        NIdentifiers::TKpId test1("");
        NIdentifiers::TKpId test2("1234");
        NIdentifiers::TKpId test3("001234");
        NIdentifiers::TKpId test4("73645sdas");

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::KINOPOISK_ID);

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), true);
        UNIT_ASSERT_EQUAL(test3.IsValid(), true);
        UNIT_ASSERT_EQUAL(test4.IsValid(), false);
    }

    Y_UNIT_TEST(TEST_yamoney_id) {
        NIdentifiers::TYamoneyId test1("");
        NIdentifiers::TYamoneyId test2("1234");
        NIdentifiers::TYamoneyId test3("001234");
        NIdentifiers::TYamoneyId test4("73645sdas");

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::YAMONEY_ID);

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), true);
        UNIT_ASSERT_EQUAL(test3.IsValid(), true);
        UNIT_ASSERT_EQUAL(test4.IsValid(), false);
    }

    Y_UNIT_TEST(TEST_xnuiqguid_id) {
        NIdentifiers::TXUniqGuid test1("");
        NIdentifiers::TXUniqGuid test2("1234");
        NIdentifiers::TXUniqGuid test3("001234");
        NIdentifiers::TXUniqGuid test4("73645sdas");

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::XUNIQ_GUID);

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), true);
        UNIT_ASSERT_EQUAL(test3.IsValid(), true);
        UNIT_ASSERT_EQUAL(test4.IsValid(), false);
    }

    Y_UNIT_TEST(TEST_avitohash) {
        NIdentifiers::TAvitoHash test1("0123456789abcdefABCDEF9876543210");  // correct
        NIdentifiers::TAvitoHash test2("0123456789abcdefABCDEF98765432100"); // not correct
        NIdentifiers::TAvitoHash test3("0123456789qbcdefABCDEF9876543210");  // not correct

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::AVITO_HASH);

        UNIT_ASSERT_EQUAL(test1.IsValid(), true);
        UNIT_ASSERT_EQUAL(test2.IsValid(), false);
        UNIT_ASSERT_EQUAL(test3.IsValid(), false);
    }

    Y_UNIT_TEST(TEST_ditid) {
        NIdentifiers::TDitId test1("0123456789abcdefABCDEF9876543210");  // correct
        NIdentifiers::TDitId test2("0123456789abcdefABCDEF98765432100"); // not correct
        NIdentifiers::TDitId test3("0123456789qbcdefABCDEF9876543210");  // not correct

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::DIT_ID);

        UNIT_ASSERT_EQUAL(test1.IsValid(), true);
        UNIT_ASSERT_EQUAL(test2.IsValid(), false);
        UNIT_ASSERT_EQUAL(test3.IsValid(), false);
    }

    Y_UNIT_TEST(Test_Tdistrui_IsValid) {
        NIdentifiers::TDistrUI test1("");                                            // not correct, empty
        NIdentifiers::TDistrUI test2("f1d2aee4-fde7-4e18-a612-4eab70dc2fcf");        // correct
        NIdentifiers::TDistrUI test3("f1d2aee4-fde7-4e18-a612-4eab70dc2Fcf");        // correct
        NIdentifiers::TDistrUI test4("f1d2aee4-fde7-4e18-a612-4eab70dc2fcfs");       // not correct, bad length
        NIdentifiers::TDistrUI test5("f1d2aee4-fde7-4e18-a612-4ea2fcff1d2-0dc2fcf"); // not correct, bad length
        NIdentifiers::TDistrUI test6("f1d2aee4-fde7-4e18-a612-4ed2-0d2fcf");         // not correct, bad length
        NIdentifiers::TDistrUI test7("f1d2aee4-0000-4e18-a612-4eab70dc2fcf");        // correct

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::DISTR_UI);

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), true);
        UNIT_ASSERT_EQUAL(test3.IsValid(), true);
        UNIT_ASSERT_EQUAL(test4.IsValid(), false);
        UNIT_ASSERT_EQUAL(test5.IsValid(), false);
        UNIT_ASSERT_EQUAL(test6.IsValid(), false);
        UNIT_ASSERT_EQUAL(test7.IsValid(), true);
    }

    Y_UNIT_TEST(Test_Tdistrui_Normalize) {
        NIdentifiers::TDistrUI test1("f1d2aee4-fde7-4e18-a612-4eab70dc2fcf");   // correct
        NIdentifiers::TDistrUI test2("f1d2aee4-fde7-4e18-a612-4eab70dc2Fcf");   // correct
        NIdentifiers::TDistrUI test3("f1d2aee4-0000-4e18-a612-4eab70dc2fcf");   // correct
        NIdentifiers::TDistrUI test4("F1D2AEE4-0000-4E18-A612-4EAB70DC2FCF");   // correct
        NIdentifiers::TDistrUI test5("f1d2aee400004e18a6124eab70dc2fcf");       // correct

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::DISTR_UI);

        UNIT_ASSERT_EQUAL(test1.Normalize(), "F1D2AEE4-FDE7-4E18-A612-4EAB70DC2FCF");
        UNIT_ASSERT_EQUAL(test2.Normalize(), "F1D2AEE4-FDE7-4E18-A612-4EAB70DC2FCF");
        UNIT_ASSERT_EQUAL(test3.Normalize(), "F1D2AEE4-0000-4E18-A612-4EAB70DC2FCF");
        UNIT_ASSERT_EQUAL(test4.Normalize(), "F1D2AEE4-0000-4E18-A612-4EAB70DC2FCF");
        UNIT_ASSERT_EQUAL(test5.Normalize(), "F1D2AEE4-0000-4E18-A612-4EAB70DC2FCF");
    }

    Y_UNIT_TEST(TEST_distrr1) {
        NIdentifiers::TDistrR1 test1("gcooiwkxalkimrbpdsthbyminovmyklvwyughafsqgiaeclyopxwmkxfpbwxcvnwxnopfgxgydvyuurquaidudjnmkshqwagopsx6b3adc20066b67696ed981d92dc60bd9");                                       // correct
        NIdentifiers::TDistrR1 test2("%ef%bf%bdadktmmoclterokdtvkiqhtgmqtjygp%7f%ef%bf%bdoehxsidyvaoongthtnxwyvelgffbbx%2b6prvgasugkwvvydrhwnroxohlndtpvwk%ef%bf%bdrjf1435d1cd30ea3edeb4b51345093%2b%ef%bf%bd212"); // not correct
        NIdentifiers::TDistrR1 test3("GCooiwkxalkimrbpdsthbyminovmyklvwyughafsqgiaeclyopxwmkxfpbwxcvnwxnopfgxgydvyuurquaidudjnmkshqwagopsx6b3adc20066b67696ed981d92dc60bd9");                                       // not correct

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::DISTR_R1);

        UNIT_ASSERT_EQUAL(test1.IsValid(), true);
        UNIT_ASSERT_EQUAL(test2.IsValid(), false);
        UNIT_ASSERT_EQUAL(test3.IsValid(), false);
    }

    Y_UNIT_TEST(TEST_partnerrecordid) {
        NIdentifiers::TPartnerRecordId test1("a101_10024");      // correct
        NIdentifiers::TPartnerRecordId test2("u2u3r2__8723g72"); // not correct
        NIdentifiers::TPartnerRecordId test3("8yf293f32:,123");  // not correct

        UNIT_ASSERT_EQUAL(test1.GetType(), EIdType::PARTNER_ID);

        UNIT_ASSERT_EQUAL(test1.IsValid(), true);
        UNIT_ASSERT_EQUAL(test2.IsValid(), true);
        UNIT_ASSERT_EQUAL(test3.IsValid(), false);
    }

    Y_UNIT_TEST(TEST_identifier_max_length) {
        TString long_email = "very";
        while (long_email.length() < 1000) {
            long_email += long_email;
        }
        long_email += "@yandex.ru";
        UNIT_ASSERT_EQUAL(long_email.length() > 1000, true);
        NIdentifiers::TEmail test1(long_email);
        NIdentifiers::TEmail test2("norm_len@yandex.ru");

        UNIT_ASSERT_EQUAL(test1.IsValid(), false);
        UNIT_ASSERT_EQUAL(test2.IsValid(), true);
    }

    Y_UNIT_TEST(TEST_has_likelihood) {
        NIdentifiers::TIdfa test_idfa("17536264-D38B-4F27-9AC2-BE63312EDFA6");
        NIdentifiers::TEmail test_email("chebureck@yandex.ru");

        UNIT_ASSERT_EQUAL(test_idfa.HasLikelihood(), true);
        UNIT_ASSERT_EQUAL(test_email.HasLikelihood(), true);

        NIdentifiers::TPhone test_phone("+7800553535");
        UNIT_ASSERT_EQUAL(test_phone.HasLikelihood(), false);
    }

    Y_UNIT_TEST(TEST_variance) {
        TVector<int> test1;
        test1.push_back(1);
        test1.push_back(1);
        test1.push_back(1);
        test1.push_back(1);

        TVector<int> test2;
        test2.push_back(1);
        test2.push_back(1);
        test2.push_back(1);
        test2.push_back(2);

        UNIT_ASSERT_DOUBLES_EQUAL(TLikelihoodTest::WrapVariance(test1, 4), 0, EPS);
        UNIT_ASSERT_DOUBLES_EQUAL(TLikelihoodTest::WrapVariance(test2, 5), 0.24, EPS);
    }

    Y_UNIT_TEST(TEST_frequencies) {
        TVector<int> test1 = TLikelihoodTest::WrapFrequencies("134");

        UNIT_ASSERT_EQUAL(test1.size(), 3);
        UNIT_ASSERT_EQUAL(test1[0], 1);
        UNIT_ASSERT_EQUAL(test1[1], 1);
        UNIT_ASSERT_EQUAL(test1[2], 1);

        TVector<int> test2 = TLikelihoodTest::WrapFrequencies("111");
        UNIT_ASSERT_EQUAL(test2.size(), 1);
        UNIT_ASSERT_EQUAL(test2[0], 3);
    }

    Y_UNIT_TEST(TEST_signs) {
        TVector<int> test1 = TLikelihoodTest::WrapSigns("1234");
        UNIT_ASSERT_EQUAL(test1.size(), 3);
        UNIT_ASSERT_EQUAL(test1[0], 1);
        UNIT_ASSERT_EQUAL(test1[0], 1);
        UNIT_ASSERT_EQUAL(test1[0], 1);

        TVector<int> test2 = TLikelihoodTest::WrapSigns("111");
        UNIT_ASSERT_EQUAL(test2.size(), 2);
        UNIT_ASSERT_EQUAL(test2[0], 0);
        UNIT_ASSERT_EQUAL(test2[0], 0);

        TVector<int> test3 = TLikelihoodTest::WrapSigns("121322");
        UNIT_ASSERT_EQUAL(test3.size(), 5);
        UNIT_ASSERT_EQUAL(test3[0], 1);
        UNIT_ASSERT_EQUAL(test3[1], -1);
        UNIT_ASSERT_EQUAL(test3[2], 1);
        UNIT_ASSERT_EQUAL(test3[3], -1);
        UNIT_ASSERT_EQUAL(test3[4], 0);
    }

    Y_UNIT_TEST(TEST_periods) {
        TVector<TString> test1 = TLikelihoodTest::WrapPeriods("123123");

        UNIT_ASSERT_EQUAL(test1.size(), 6);
        UNIT_ASSERT_EQUAL(test1[0], "123123");
        UNIT_ASSERT_EQUAL(test1[1]  + " " + test1[2], "132 213");
        UNIT_ASSERT_EQUAL(test1[3] + " " + test1[4] + " " + test1[5], "11 22 33");
    }

    Y_UNIT_TEST(TEST_substrings) {
        TMap<TString, int> test1 = TLikelihoodTest::WrapSubstringsRepetitions("123123");

        UNIT_ASSERT_EQUAL(test1["123"], 2);
        UNIT_ASSERT_EQUAL(test1["1231"], 1);
    }

    Y_UNIT_TEST(TEST_get_likelihood_funcs) {
        NIdentifiers::TIdfa test_idfa("17536264-D38B-4F27-9AC2-BE63312EDFA6");

        UNIT_ASSERT_EQUAL(test_idfa.GetFrequencySpread() != ABSENT_LIKELIHOOD, true);
        UNIT_ASSERT_EQUAL(test_idfa.GetAlternating() != ABSENT_LIKELIHOOD, true);
        UNIT_ASSERT_EQUAL(test_idfa.GetPeriodicity() != ABSENT_LIKELIHOOD, true);
        UNIT_ASSERT_EQUAL(test_idfa.GetRepetitions() != ABSENT_LIKELIHOOD, true);

        NIdentifiers::TEmail test_email("qwerty@yandex.ru");

        UNIT_ASSERT_EQUAL(test_email.GetFrequencySpread() != ABSENT_LIKELIHOOD, true);
        UNIT_ASSERT_EQUAL(test_email.GetAlternating() != ABSENT_LIKELIHOOD, true);
        UNIT_ASSERT_EQUAL(test_email.GetPeriodicity() != ABSENT_LIKELIHOOD, true);
        UNIT_ASSERT_EQUAL(test_email.GetRepetitions() != ABSENT_LIKELIHOOD, true);
    }

    Y_UNIT_TEST(TEST_max_login_len) {
        NIdentifiers::TEmail long_local("1234567890123456789012345678901234567890123456789012345678901234+x@example.com");
        UNIT_ASSERT_EQUAL(long_local.IsValid(), false);

        // 256 characters
        NIdentifiers::TEmail long_domain("a@skjdhfgfksdjhfgkdfhgkdfjhgkdfhgkdfjhgkdfhgkdfhgkdfhgkdfhgjdfghdkfjghkdjhfgkdhgkdfhgkdfhgkdhfkghkdfjhgkdfhgdkfghkdfjghkdfjhgkdjfhgkdjfhgkdfjhgkdfhgkdfhgkdfhgkdfjhgkdfjhgkdfjhgkdfhgkdfjhgkdfjhgkdfjhgkdfjhgkdfhgkfdhgkdfjhgkdfhgkdfjhgdfkgjhdkfghkdfjhgfkdjgh.ru");
        UNIT_ASSERT_EQUAL(long_domain.IsValid(), false);
    }

    Y_UNIT_TEST(TEST_invalid_identifier_by_name) {
        // huawai id is incorect
        UNIT_ASSERT_EXCEPTION_CONTAINS(NIdentifiers::TGenericID("HuID", "f1d2aee4-fde7-4e18-a612-4eab70dc2fcf"),
                                       yexception, "Identifier 'huid' not found");
        // open id is correct
        UNIT_ASSERT_EQUAL(NIdentifiers::TGenericID("OaID", "f1d2aee4-fde7-4e18-a612-4eab70dc2fcf").GetTypeString(), "oaid");
    }

    Y_UNIT_TEST(TEST_ssp_user_id_normalize) {
        NIdentifiers::TSspUserId test_ssp(12345, "abc");
        UNIT_ASSERT_UNEQUAL(test_ssp.Normalize(), "12345@abc");
        UNIT_ASSERT_EQUAL(test_ssp.Normalize(), "12345@ABC");
    }

    Y_UNIT_TEST(Test_ysclid_smoke) {
        using namespace NIdentifiers;

        TString oldok{"ku7f6hp5xd"};
        TString newok{"l294wkcyaa"};
        TString empty{""};
        TString shorter{"ksil7dw"};
        TString wrong{"k@il7dwgjk"};

        TString newformatoldok{"ku7f6hp5xd0"};
        TString newformatnewok{"l294wkcyaa123432312"};
        TString newformattoolong{"l294wkcyaa12345678901"};
        TString newformatnotnumbers{"l294wkcyaa123456axdfg"};

        UNIT_ASSERT(TYSClid::Validate(oldok));
        UNIT_ASSERT(TYSClid::Validate(newok));
        UNIT_ASSERT(!TYSClid::Validate(empty));
        UNIT_ASSERT(!TYSClid::Validate(shorter));
        UNIT_ASSERT(!TYSClid::Validate(wrong));

        UNIT_ASSERT(TYSClid::Validate(newformatoldok));
        UNIT_ASSERT(TYSClid::Validate(newformatnewok));
        UNIT_ASSERT(!TYSClid::Validate(newformattoolong));
        UNIT_ASSERT(!TYSClid::Validate(newformatnotnumbers));

        UNIT_ASSERT_EQUAL(TYSClid::GetTimestamp(oldok), 1633035512);
        UNIT_ASSERT_EQUAL(TYSClid::GetTimestamp(newok), 1650553444);
        UNIT_ASSERT_EXCEPTION_CONTAINS(TYSClid::GetTimestamp(wrong), yexception, "valid");

        UNIT_ASSERT_EQUAL(TYSClid::GetTimestamp(newformatoldok), 1633035512);
        UNIT_ASSERT_EQUAL(TYSClid::GetTimestamp(newformatnewok), 1650553444);
        UNIT_ASSERT_EXCEPTION_CONTAINS(TYSClid::GetTimestamp(newformattoolong), yexception, "valid");

        UNIT_ASSERT(!TYSClid::Significate(oldok));
        UNIT_ASSERT(TYSClid::Significate(newok));
        UNIT_ASSERT(!TYSClid::Significate(newformatoldok));
        UNIT_ASSERT(TYSClid::Significate(newformatnewok));

        UNIT_ASSERT(TYSClid::Significate(TYSClid::Next()));
    }

    Y_UNIT_TEST(Test_domain_validation) {
        using namespace NIdentifiers;

        const TVector<TString> goodDomains{
            "yandex123.ru", "xn--e1ajeds9e.xn--p1ai", "yandex.ru.", "yandex.yandexyandex",
            "yan--dex.ru", "www.ya----ndex.r-----u", "www.Disk.y4nd3x.RU", "a3.com",
            "www.wonderful.so.many.subdomains.in.this.pretty.domain.name.yandex.az",
            "aa.aa",
        };

        const TVector<TString> badDomains{
            "-yandex.ru","yandex.-ru",".com", "com", ".a", ".", "..", ".az.", "yandex-.yandex.ru", "yandex..ru", "",
            "yandex.ru..", ".yandex.ru", "..yandex.ru", "3andex.ru", "yan_dex.ru", "y@n.aa", "@asdasd.com", "aa.a",
            "wowwhatanincrediblylongpartofthedomainnamewhichisvalidbutcontainsmorethen63symbolsinit.com", "3a.com"
        };

        for (const auto& domain : goodDomains) {
            UNIT_ASSERT(THostname::Validate(domain));
        }

        for (const auto& domain : badDomains) {
            UNIT_ASSERT(!THostname::Validate(domain));
        }

        UNIT_ASSERT(THostname::Validate(THostname::Next()));
    }

    Y_UNIT_TEST(Test_domain_signification) {
        using namespace NIdentifiers;

        const TVector<TString> significantDomains{
            "yandex123.ru", "xn--e1ajeds9e.xn--p1ai", "example.com.tr",
        };

        const TVector<TString> insignificantDomains{
            "example.net", "yandex.test", "some.invalid.domai.n", "",
            "localhost.localdomain",
        };

        for (const auto& domain : significantDomains) {
            UNIT_ASSERT(THostname::Significate(domain));
        }

        for (const auto& domain : insignificantDomains) {
            UNIT_ASSERT(!THostname::Significate(domain));
        }
    }

    Y_UNIT_TEST(Test_isyandex) {
        using namespace NIdentifiers;

        UNIT_ASSERT(THostname::IsYandex("yandex.ru"));
        UNIT_ASSERT(THostname::IsYandex("ya.ru"));
        UNIT_ASSERT(THostname::IsYandex("www.ya.ru"));
        UNIT_ASSERT(THostname::IsYandex("www.yandex.com.tr"));
        UNIT_ASSERT(!THostname::IsYandex("www.yandex.com.to"));
        UNIT_ASSERT(THostname::IsYandex("www.yandex.com"));
        UNIT_ASSERT(!THostname::IsYandex(".yandex.ru"));
        UNIT_ASSERT(!THostname::IsYandex("myandex.ru"));
        UNIT_ASSERT(THostname::IsYandex("disk.yandex.ru"));
        UNIT_ASSERT(!THostname::IsYandex("www.yandex.ru.evil.com"));
    }

    Y_UNIT_TEST(Test_isyastatic) {
        using namespace NIdentifiers;

        UNIT_ASSERT(THostname::IsYastatic("yastatic.net"));
        UNIT_ASSERT(!THostname::IsYastatic(".yastatic.net"));
        UNIT_ASSERT(!THostname::IsYastatic("myastatic.net"));
        UNIT_ASSERT(!THostname::IsYastatic("file.at.yastatic.net"));
        UNIT_ASSERT(!THostname::IsYastatic("www.yastatic.net.evil.com"));
    }

    Y_UNIT_TEST(TEST_normalized_equality) {
        {
            const NIdentifiers::TGaid lowerCased{"fc10dd733d50481c910974234ed69642"};
            const NIdentifiers::TGaid upperCased{"FC10DD733D50481C910974234ED69642"};

            UNIT_ASSERT(lowerCased == upperCased);
        }

        {
            const NIdentifiers::TEmail withVERP{"test.email+facebook@some.domain.com"};
            const NIdentifiers::TEmail withoutVERP{"test.email@some.domain.com"};

            UNIT_ASSERT(withVERP == withoutVERP);
        }

        {
            const NIdentifiers::THostname upperCased{"YanDex.ru."};
            const NIdentifiers::THostname andDotted{"YanDex.ru."};
            const NIdentifiers::THostname plain{"yandex.ru"};
            const NIdentifiers::THostname empty{""};

            UNIT_ASSERT(upperCased == plain);
            UNIT_ASSERT(andDotted == upperCased);
            UNIT_ASSERT(empty != plain);
        }

    }
}
