#include <iosfwd>

#include <crypta/lib/native/identifiers/lib/generic.h>
#include <crypta/lib/native/identifiers/lib/id_types/all.h>
#include <library/cpp/testing/unittest/registar.h>

template <class TypeIdentifier>
void IdentifierTest(TString bad_prefix) {
    TString prev;
    for (size_t index{0}; index < 10; ++index) {
        const TString next{TypeIdentifier::Next()};
        UNIT_ASSERT(prev != next);
        UNIT_ASSERT(next.size() > 1);
        UNIT_ASSERT(TypeIdentifier::Validate(next));
        UNIT_ASSERT(!TypeIdentifier::Validate(bad_prefix + next));

        UNIT_ASSERT(TypeIdentifier(next).IsValid());
        UNIT_ASSERT(TypeIdentifier(next).IsSignificant());

        UNIT_ASSERT(!TypeIdentifier(bad_prefix + next).IsValid());
        UNIT_ASSERT(!TypeIdentifier(bad_prefix + next).IsSignificant());

        prev = next;
    }
}

template <class TypeIdentifier>
void GenericIdentifierTest(TString bad_prefix) {
    TString prev;
    const auto proto_type{TypeIdentifier("").GetType()};
    for (size_t index{0}; index < 10; ++index) {
        const TString next{NIdentifiers::TGenericID::Next(proto_type)};
        UNIT_ASSERT(prev != next);
        UNIT_ASSERT(next.size() > 1);

        UNIT_ASSERT(TypeIdentifier::Validate(next));
        UNIT_ASSERT(TypeIdentifier::Significate(next));

        UNIT_ASSERT(NIdentifiers::TGenericID(proto_type, next).IsValid());
        UNIT_ASSERT(NIdentifiers::TGenericID(proto_type, next).IsSignificant());

        UNIT_ASSERT(!TypeIdentifier::Validate(bad_prefix + next));
        UNIT_ASSERT(!TypeIdentifier::Significate(bad_prefix + next));

        UNIT_ASSERT(!NIdentifiers::TGenericID(proto_type, bad_prefix + next).IsValid());
        UNIT_ASSERT(!NIdentifiers::TGenericID(proto_type, bad_prefix + next).IsSignificant());

        prev = next;
    }
}

Y_UNIT_TEST_SUITE(TestStaticIdentifier) {

    Y_UNIT_TEST(TEST_email_static_valid) {
        TString prev;
        for (size_t index{0}; index < 10; ++index) {
            const TString next{NIdentifiers::TEmail::Next()};
            UNIT_ASSERT(next.size() > 0);
            UNIT_ASSERT(next.size() < 25);
            UNIT_ASSERT(next.find('@') >= 5);
            UNIT_ASSERT(next.find('@') <= 10);
            UNIT_ASSERT(next != prev);
            UNIT_ASSERT(NIdentifiers::TEmail::Validate(next));
            UNIT_ASSERT(!NIdentifiers::TEmail::Validate("bad@" + next));
            prev = next;
        }
    }

    Y_UNIT_TEST(TEST_ssp_user_id_static_valid) {
        TString prev;
        for (size_t index{0}; index < 10; ++index) {
            const TString next{NIdentifiers::TSspUserId::Next()};
            UNIT_ASSERT(next.size() > 0);
            UNIT_ASSERT(next != prev);
            UNIT_ASSERT(NIdentifiers::TSspUserId::Validate(next));
            prev = next;
        }
        UNIT_ASSERT(!NIdentifiers::TSspUserId::Validate(""));
        UNIT_ASSERT(!NIdentifiers::TSspUserId::Validate("@a"));
        UNIT_ASSERT(!NIdentifiers::TSspUserId::Validate("abc@a"));
        UNIT_ASSERT(!NIdentifiers::TSspUserId::Validate("10001@"));
        UNIT_ASSERT(!NIdentifiers::TSspUserId::Validate("10001@a "));
        UNIT_ASSERT(!NIdentifiers::TSspUserId::Validate("10001@a b"));
        UNIT_ASSERT(NIdentifiers::TSspUserId::Validate("0@0"));
        UNIT_ASSERT(NIdentifiers::TSspUserId::Validate("10000@a"));
        UNIT_ASSERT(NIdentifiers::TSspUserId::Validate("10001@a"));
        UNIT_ASSERT(NIdentifiers::TSspUserId::Validate("10001@ABC"));

        UNIT_ASSERT(!NIdentifiers::TSspUserId::Significate("0@0"));
        UNIT_ASSERT(!NIdentifiers::TSspUserId::Significate("10000@a"));
        UNIT_ASSERT(NIdentifiers::TSspUserId::Significate("10001@a"));
        UNIT_ASSERT(NIdentifiers::TSspUserId::Significate("10001@ABC"));

    }

    Y_UNIT_TEST(TEST_yuid_static_valid) {
        IdentifierTest<NIdentifiers::TYandexuid>("-1");
    }

    Y_UNIT_TEST(TEST_icookie_static_valid) {
        IdentifierTest<NIdentifiers::TIcookie>("-1");
    }

    Y_UNIT_TEST(TEST_fb_id_static_valid) {
        IdentifierTest<NIdentifiers::TFbId>("z");
    }

    Y_UNIT_TEST(TEST_vk_id_static_valid) {
        IdentifierTest<NIdentifiers::TVkId>("~vk");
    }

    Y_UNIT_TEST(TEST_avito_hash_static_valid) {
        IdentifierTest<NIdentifiers::TAvitoHash>("0");
    }

    Y_UNIT_TEST(TEST_avito_id_static_valid) {
        IdentifierTest<NIdentifiers::TAvitoId>("-1");
    }

    Y_UNIT_TEST(TEST_crypta_id_static_valid) {
        IdentifierTest<NIdentifiers::TCryptaId>("-1");
    }

    Y_UNIT_TEST(TEST_distr_r1_static_valid) {
        IdentifierTest<NIdentifiers::TDistrR1>("0");
    }

    Y_UNIT_TEST(TEST_distr_ui_static_valid) {
        IdentifierTest<NIdentifiers::TDistrUI>("0");
    }

    Y_UNIT_TEST(TEST_dit_id_static_valid) {
        IdentifierTest<NIdentifiers::TDitId>("0");
    }

    Y_UNIT_TEST(TEST_duid_static_valid) {
        IdentifierTest<NIdentifiers::TDuid>("-");
    }

    Y_UNIT_TEST(TEST_gaid_static_valid) {
        IdentifierTest<NIdentifiers::TGaid>("0");
    }

    Y_UNIT_TEST(TEST_idfa_static_valid) {
        IdentifierTest<NIdentifiers::TIdfa>("0");
    }

    Y_UNIT_TEST(TEST_ifv_static_valid) {
        IdentifierTest<NIdentifiers::TIfv>("0");
    }

    Y_UNIT_TEST(TEST_idfa_gaid_static_valid) {
        IdentifierTest<NIdentifiers::TIdfaGaid>("0");
    }

    Y_UNIT_TEST(TEST_kp_id_static_valid) {
        IdentifierTest<NIdentifiers::TKpId>("-0");
    }

    Y_UNIT_TEST(TEST_login_static_valid) {
        TString prev;
        for (size_t index = 0; index < 10; ++index) {
            TString next = NIdentifiers::TLogin::Next();
            UNIT_ASSERT(next.size() > 0);
            UNIT_ASSERT(next.size() < 25);
            UNIT_ASSERT(next.find('@') == TString::npos);
            UNIT_ASSERT(next != prev);
            UNIT_ASSERT(NIdentifiers::TLogin::Validate(next));
            UNIT_ASSERT(!NIdentifiers::TLogin::Validate("phne--e" + next));
            prev = next;
        }
    }

    Y_UNIT_TEST(TEST_mac_static_valid) {
        IdentifierTest<NIdentifiers::TMac>("0");
    }

    Y_UNIT_TEST(TEST_md5_static_valid) {
        IdentifierTest<NIdentifiers::TMd5>("0");
    }

    Y_UNIT_TEST(TEST_email_md5_static_valid) {
        IdentifierTest<NIdentifiers::TEmailMd5>("0");
    }

    Y_UNIT_TEST(TEST_phone_md5_static_valid) {
        IdentifierTest<NIdentifiers::TPhoneMd5>("0");
    }

    Y_UNIT_TEST(TEST_macExt_static_valid) {
        IdentifierTest<NIdentifiers::TMacExt>("0");
    }

    Y_UNIT_TEST(TEST_macExt_md5_static_valid) {
        IdentifierTest<NIdentifiers::TMacExtMd5>("0");
    }

    Y_UNIT_TEST(TEST_mm_device_id_static_valid) {
        IdentifierTest<NIdentifiers::TMmDeviceId>("0");
    }

    Y_UNIT_TEST(TEST_ok_id_static_valid) {
        IdentifierTest<NIdentifiers::TOkId>("-1");
    }

    Y_UNIT_TEST(TEST_partner_record_id_static_valid) {
        IdentifierTest<NIdentifiers::TPartnerRecordId>("!");
    }

    Y_UNIT_TEST(TEST_phone_static_valid) {
        IdentifierTest<NIdentifiers::TPhone>("0");
    }

    Y_UNIT_TEST(TEST_puid_static_valid) {
        IdentifierTest<NIdentifiers::TPuid>("-");
    }

    Y_UNIT_TEST(TEST_sha256_static_valid) {
        IdentifierTest<NIdentifiers::TSha256>("0");
    }

    Y_UNIT_TEST(TEST_email_sha256_static_valid) {
        IdentifierTest<NIdentifiers::TEmailSha256>("0");
    }

    Y_UNIT_TEST(TEST_phone_sha256_static_valid) {
        IdentifierTest<NIdentifiers::TPhoneSha256>("0");
    }

    Y_UNIT_TEST(TEST_uuid_static_valid) {
        IdentifierTest<NIdentifiers::TUuid>("0");
    }

    Y_UNIT_TEST(TEST_edadeal_static_valid) {
        IdentifierTest<NIdentifiers::TEdadealUid>("0");
    }

    Y_UNIT_TEST(TEST_xuniq_guid_static_valid) {
        IdentifierTest<NIdentifiers::TXUniqGuid>("-1");
    }

    Y_UNIT_TEST(TEST_yamoney_id_static_valid) {
        IdentifierTest<NIdentifiers::TYamoneyId>("-1");
    }

    Y_UNIT_TEST(TEST_mm_dev_id_hash_static_valid) {
        IdentifierTest<NIdentifiers::TMmDeviceIdHash>("-1");
    }

    Y_UNIT_TEST(TEST_android_id_static_valid) {
        IdentifierTest<NIdentifiers::TAndroidId>("0");
    }

    Y_UNIT_TEST(TEST_imei_static_valid) {
        IdentifierTest<NIdentifiers::TImei>("t");
    }

    Y_UNIT_TEST(TEST_generic_id_static_valid) {
        GenericIdentifierTest<NIdentifiers::TGaid>("-1");
        GenericIdentifierTest<NIdentifiers::TIdfa>("-1");
        GenericIdentifierTest<NIdentifiers::TIfv>("-1");
        GenericIdentifierTest<NIdentifiers::TKpId>("-1");
        GenericIdentifierTest<NIdentifiers::TMac>("-1");
        GenericIdentifierTest<NIdentifiers::TYamoneyId>("-1");
        GenericIdentifierTest<NIdentifiers::TYandexuid>("-1");
    }

    Y_UNIT_TEST(TEST_Significant_DeviceId) {
        UNIT_ASSERT(NIdentifiers::TMmDeviceId("06300C54-22A4-4138-B267-880DBA124038").IsSignificant());
        UNIT_ASSERT(NIdentifiers::TGaid("06300C54-22A4-4138-B267-880DBA124038").IsSignificant());
        UNIT_ASSERT(NIdentifiers::TIdfa("06300C54-22A4-4138-B267-880DBA124038").IsSignificant());
        UNIT_ASSERT(NIdentifiers::TOaid("06300C54-22A4-4138-B267-880DBA124038").IsSignificant());
        UNIT_ASSERT(NIdentifiers::TIfv("06300C54-22A4-4138-B267-880DBA124038").IsSignificant());
        UNIT_ASSERT(NIdentifiers::TUuid("06300c5422a44138b267880dba124038").IsSignificant());
        UNIT_ASSERT(!NIdentifiers::TMmDeviceId("00000000-0000-0000-0000-000000000000").IsSignificant());
        UNIT_ASSERT(!NIdentifiers::TGaid("00000000-0000-0000-0000-000000000000").IsSignificant());
        UNIT_ASSERT(!NIdentifiers::TIdfa("00000000-0000-0000-0000-000000000000").IsSignificant());
        UNIT_ASSERT(!NIdentifiers::TOaid("00000000-0000-0000-0000-000000000000").IsSignificant());
        UNIT_ASSERT(!NIdentifiers::TIfv("00000000-0000-0000-0000-000000000000").IsSignificant());
        UNIT_ASSERT(!NIdentifiers::TUuid("00000000000000000000000000000000").IsSignificant());
        UNIT_ASSERT(!NIdentifiers::TMmDeviceId("11111111-1111-1111-1111-111111111111").IsSignificant());
        UNIT_ASSERT(!NIdentifiers::TGaid("55555555-5555-5555-5555-555555555555").IsSignificant());
        UNIT_ASSERT(!NIdentifiers::TIdfa("77777777-7777-7777-7777-777777777777").IsSignificant());
        UNIT_ASSERT(!NIdentifiers::TOaid("AAAAAAAA-AAAA-AAAA-AAAA-AAAAAAAAAAAA").IsSignificant());
        UNIT_ASSERT(!NIdentifiers::TIfv("FFFFFFFF-FFFF-FFFF-FFFF-FFFFFFFFFFFF").IsSignificant());
        UNIT_ASSERT(!NIdentifiers::TUuid("99999999999999999999999999999999").IsSignificant());
    }
}
