#include "ip_filter.h"

#include <util/string/strip.h>

namespace {
    TIpv6Address NormalizeSafe(const TStringBuf addrStr, bool& ok) {
        auto addr = TIpv6Address::FromString(addrStr, ok);
        if (!ok) {
            return TIpv6Address();
        }

        if(!addr.Isv4MappedTov6()) {
            return addr;
        }

        return addr.TryToExtractIpv4From6();
    }

    TMaybe<TIpAddressRange> TryFromRangeString(const TStringBuf rangeString) {
        auto idx = rangeString.find('-');
        if (idx == TString::npos) {
            return Nothing();
        }

        TStringBuf start, end;
        rangeString.SplitAt(idx, start, end);
        end.Skip(1);

        bool ok;
        const auto from = NormalizeSafe(StripString(start), ok);
        if (!ok) {
            return Nothing();
        }

        const auto to = NormalizeSafe(StripString(end), ok);
        if (!ok) {
            return Nothing();
        }

        return TIpAddressRange::From(from).To(to).Build();
    }
}

namespace NCrypta {
    TIpAddressRange TIpFilter::MakeRange(const TString& rangeString) {
        auto range = TIpAddressRange::TryFromCidrString(rangeString);
        if (range) {
            return *range;
        }

        range = TryFromRangeString(rangeString);
        if (range) {
            return *range;
        }

        ythrow yexception() << "Malformed ip range description: '" << rangeString << "'";
    }

    bool TIpFilter::IsPassing(const TStringBuf ipStr) const {
        if (Ranges.empty() || ipStr.empty()) {
            return false;
        }

        bool ok;
        auto ip = NormalizeSafe(ipStr, ok);
        if (!ok) {
            return false;
        }

        auto candidate = Ranges.lower_bound(ip);

        if (candidate != Ranges.end() && candidate->first == ip) {
            return true;
        }
        if (candidate != Ranges.begin()) {
            return prev(candidate)->second.Contains(ip);
        }

        return false;
    }
}
